/**
 *  Simple MPEG parser to achieve network/service information.
 *
 *  refered standards:
 *
 *    ETSI EN 300 468
 *    ETSI TR 101 211
 *    ETSI ETR 211
 *    ITU-T H.222.0
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/poll.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>

#include <linux/dvb/frontend.h>
#include <linux/dvb/dmx.h>

#include "list.h"
#include "diseqc.h"
#include "dump-zap.h"
#include "dump-vdr.h"
#include "scan.h"


static char demux_devname[80];

static void dump_dvb_parameters (FILE *f, fe_type_t type, struct dvb_frontend_parameters *p);

static int long_timeout;
static int current_tp_only;

enum table_type {
	PAT,
	PMT,
	SDT,
	NIT
};


struct list_entry {
	int id1;
	int id2;
};


struct list {
	char *name;
	struct list_entry *entry;
	int count;
	int entry_size;
};


struct network {
	int dummy;
	int network_id;
	char *network_name;
};

enum format {
        OUTPUT_ZAP,
        OUTPUT_VDR,
	OUTPUT_PIDS
};
static enum format output_format = OUTPUT_ZAP;


enum polarisation {
	POLARISATION_HORIZONTAL     = 0x00,
	POLARISATION_VERTICAL       = 0x01,
	POLARISATION_CIRCULAR_LEFT  = 0x02,
	POLARISATION_CIRCULAR_RIGHT = 0x03
};

struct transponder {
	int network_id;
	int transport_stream_id;
	fe_type_t type;
	struct dvb_frontend_parameters param;
	enum polarisation polarisation : 2;	   /*  only for DVB-S */
	int scan_done		  : 1;
	int last_tuning_failed	  : 1;
	int other_frequency_flag  : 1;		/* DVB-T */
	int n_other_f;
	uint32_t *other_f;	/* DVB-T freqeuency-list descriptor */
};


enum running_mode {
	RM_NOT_RUNNING = 0x01,
	RM_STARTS_SOON = 0x02,
	RM_PAUSING     = 0x03,
	RM_RUNNING     = 0x04
};

#define AUDIO_CHAN_MAX (5)

struct service {
	int transport_stream_id;
	int service_id;
	char *provider_name;
	char *service_name;
	uint16_t pmt_pid;
	uint16_t pcr_pid;
	uint16_t video_pid;
	uint16_t audio_pid[AUDIO_CHAN_MAX];
	char audio_lang[AUDIO_CHAN_MAX][4];
        int audio_num;
	uint16_t teletext_pid;
	uint16_t subtitling_pid;
	uint16_t ac3_pid;
	int type                  : 8;
	int scrambled	          : 1;
	enum running_mode running : 3;
	void *priv;
};

struct section_buf {
	struct list_head list_head;
	const char *dmx_devname;
	int run_once;
	int fd;
	int pid;
	int table_id [3];
	int id [3];
	int section_version_number [3];
	uint8_t section_done [3][32];
	int sectionfilter_done [3];
	unsigned char buf [1024];
	time_t timeout;
	time_t start_time;
	time_t running_time;
};



static
struct list network_list = {
	"network_list",
	NULL,
	0,
	sizeof(struct network)
};

static
struct list transponder_list = {
	"transponder_list",
	NULL,
	0,
	sizeof(struct transponder)
};

static
struct list service_list = {
	"service_list",
	NULL,
	0,
	sizeof(struct service)
};

static
void setup_filter (struct section_buf* s, const char *dmx_devname,
		   int pid, int tid0, int tid1, int tid2,
		   int run_once, int timeout);
void add_filter (struct section_buf *s);


static
void* find_entry_by_id (struct list *list, int id1, int id2)
{
	int i;
	long entry = (long) list->entry;

	for (i=0; i<list->count; i++, entry += list->entry_size) {
		struct list_entry *e = (struct list_entry*) entry;

		if (e->id1 == id1 && (e->id2 == id2 || id2 == -1))
			return e;

		if (e->id2 == id2 && id1 == -1)
			return e;

		if (e->id1 == id1 && e->id2 == -1) {
			e->id2 = id2;
			return e;
		}

		if (e->id1 == -1 && e->id2 == id2) {
			e->id1 = id1;
			return e;
		}
	}

	return NULL;
}


static
void* find_or_alloc_entry_by_id (struct list *list, int id1, int id2)
{
	struct list_entry *e = find_entry_by_id (list, id1, id2);
	long addr;

	if (!e) {
		list->entry = realloc (list->entry,
				       (list->count+1) * list->entry_size);

		addr = (long) list->entry + list->count * list->entry_size;
		e = (struct list_entry*) addr;

		memset (e, 0, list->entry_size);

		e->id1 = id1;
		e->id2 = id2;
		list->count++;
	}

	return e;
}


static
void* find_nth_entry (struct list *list, unsigned int n)
{
	if (n >= list->count)
		return NULL;

	return (void*) ((long) list->entry + n * list->entry_size);
}


static
void parse_iso639_language_descriptor (const unsigned char *buf, struct service *s)
{
	unsigned char len = buf [1];

	buf += 2;

	if (len >= 4) {
		//MSG("    LANG=%.3s %d", buf, buf[3]);
		memcpy(s->audio_lang[s->audio_num], buf, 3);
#if 0
		/* seems like the audio_type is wrong all over the place */
		//if (buf[3] == 0) -> normal
		if (buf[3] == 1)
			s->audio_lang[s->audio_num][3] = '!'; /* clean effects (no language) */
		else if (buf[3] == 2)
			s->audio_lang[s->audio_num][3] = '?'; /* for the hearing impaired */
		else if (buf[3] == 3)
			s->audio_lang[s->audio_num][3] = '+'; /* visually impaired commentary */
#endif
	}
}

static
void parse_network_name_descriptor (const unsigned char *buf, struct network *n)
{
	unsigned char len = buf [1];

	n->network_name = malloc (len + 1);
	memcpy (n->network_name, buf + 2, len);
	n->network_name[len] = '\0';
	MSG("(%s)", n->network_name);
}


static
long bcd32_to_cpu (const int b0, const int b1, const int b2, const int b3)
{
	return ((b0 >> 4) & 0x0f) * 10000000 + (b0 & 0x0f) * 1000000 +
	       ((b1 >> 4) & 0x0f) * 100000   + (b1 & 0x0f) * 10000 +
	       ((b2 >> 4) & 0x0f) * 1000     + (b2 & 0x0f) * 100 +
	       ((b3 >> 4) & 0x0f) * 10       + (b3 & 0x0f);
}


static const fe_code_rate_t fec_tab [8] = {
	FEC_AUTO, FEC_1_2, FEC_2_3, FEC_3_4,
	FEC_5_6, FEC_7_8, FEC_NONE, FEC_NONE
};


static const fe_modulation_t qam_tab [6] = {
	QAM_AUTO, QAM_16, QAM_32, QAM_64, QAM_128, QAM_256
};


static
void parse_cable_delivery_system_descriptor (const unsigned char *buf,
					     struct transponder *t)
{
	t->type = FE_QAM;

	t->param.frequency = bcd32_to_cpu (buf[2], buf[3], buf[4], buf[5]);
	t->param.frequency *= 100;
	t->param.u.qam.fec_inner = fec_tab[buf[12] & 0x07];
	t->param.u.qam.symbol_rate = 10 * bcd32_to_cpu (buf[9],
							buf[10],
							buf[11],
							buf[12] & 0xf0);
	if ((buf[8] & 0x0f) > 5)
		t->param.u.qam.modulation = QAM_AUTO;
	else
		t->param.u.qam.modulation = qam_tab[buf[8] & 0x0f];

	fprintf (stderr, "%#06x/%#06x ", t->network_id, t->transport_stream_id);
	dump_dvb_parameters (stderr, t->type, &t->param);
	if (t->scan_done)
		fprintf (stderr, " (done)");
	if (t->last_tuning_failed)
		fprintf (stderr, " (tuning failed)");
	fprintf (stderr, "\n");
}


static
void parse_satellite_delivery_system_descriptor (const unsigned char *buf,
						 struct transponder *t)
{
	t->type = FE_QPSK;
	t->param.frequency = 10 * bcd32_to_cpu (buf[2], buf[3], buf[4], buf[5]);
	t->param.u.qpsk.fec_inner = fec_tab[buf[12] & 0x07];
	t->param.u.qpsk.symbol_rate = 10 * bcd32_to_cpu (buf[9],
							 buf[10],
							 buf[11],
							 buf[12] & 0xf0);

	t->polarisation = (buf[8] >> 5) & 0x03;

	dump_dvb_parameters (stderr, t->type, &t->param);
	if (t->scan_done)
		fprintf (stderr, " (done)");
	if (t->last_tuning_failed)
		fprintf (stderr, " (tuning failed)");
	fprintf (stderr, "\n");
}


static
void parse_terrestrial_delivery_system_descriptor (const unsigned char *buf,
						   struct transponder *t)
{
	static const fe_modulation_t m_tab [] = { QPSK, QAM_16, QAM_64, QAM_AUTO };
	static const fe_code_rate_t ofec_tab [8] = { FEC_1_2, FEC_2_3, FEC_3_4,
					       FEC_5_6, FEC_7_8 };
	struct dvb_ofdm_parameters *o = &t->param.u.ofdm;

	t->type = FE_OFDM;

	t->param.frequency = (buf[2] << 24) | (buf[3] << 16);
	t->param.frequency |= (buf[4] << 8) | buf[5];
	t->param.frequency *= 10;

	o->bandwidth = BANDWIDTH_8_MHZ + ((buf[6] >> 5) & 0x3);
	o->constellation = m_tab[(buf[7] >> 6) & 0x3];
	o->hierarchy_information = HIERARCHY_NONE + ((buf[7] >> 3) & 0x3);

	if ((buf[7] & 0x7) > 4)
		o->code_rate_HP = FEC_AUTO;
	else
		o->code_rate_HP = ofec_tab [buf[7] & 0x7];

	if (((buf[8] >> 5) & 0x7) > 4)
		o->code_rate_HP = FEC_AUTO;
	else
		o->code_rate_HP = ofec_tab [(buf[8] >> 5) & 0x7];

	o->guard_interval = GUARD_INTERVAL_1_32 + ((buf[8] >> 3) & 0x3);

	o->transmission_mode = (buf[8] & 0x2) ?
			       TRANSMISSION_MODE_8K :
			       TRANSMISSION_MODE_2K;

	t->other_frequency_flag = (buf[8] & 0x01);

	dump_dvb_parameters (stderr, t->type, &t->param);
	if (t->scan_done)
		fprintf (stderr, " (done)");
	if (t->last_tuning_failed)
		fprintf (stderr, " (tuning failed)");
	fprintf (stderr, "\n");
}

static
void parse_frequency_list_descriptor (const unsigned char *buf,
				      struct transponder *t)
{
	int n, i;
	typeof(*t->other_f) f;

	if (t->other_f)
		return;

	n = (buf[1] - 1) / 4;
	if (n < 1 || (buf[2] & 0x03) != 3)
		return;

	t->other_f = calloc(n, sizeof(*t->other_f));
	t->n_other_f = n;
	buf += 3;
	for (i = 0; i < n; i++) {
		f = (buf[0] << 24) | (buf[1] << 16) | (buf[2] << 8) | buf[3];
		t->other_f[i] = f * 10;
		buf += 4;
	}
}

static
void parse_service_descriptor (const unsigned char *buf, struct service *s)
{
	unsigned char len;
	unsigned char *src, *dest;

	s->type = buf[2];

	buf += 3;
	len = *buf;
	buf++;

	if (s->provider_name)
		free (s->provider_name);

	s->provider_name = malloc (len + 1);
	memcpy (s->provider_name, buf, len);
	s->provider_name[len] = '\0';

	/* remove control characters (FIXME: handle short/long name) */
	/* FIXME: handle character set correctly (e.g. via iconv)
	 * c.f. EN 300 468 annex A */
	for (src = dest = s->provider_name; *src; src++)
		if (*src >= 0x20 && (*src < 0x80 || *src > 0x9f))
			*dest++ = *src;
	*dest = '\0';
	if (!s->provider_name[0]) {
		/* zap zero length names */
		free (s->provider_name);
		s->provider_name = 0;
	}

	if (s->service_name)
		free (s->service_name);

	buf += len;
	len = *buf;
	buf++;

	s->service_name = malloc (len + 1);
	memcpy (s->service_name, buf, len);
	s->service_name[len] = '\0';

	/* remove control characters (FIXME: handle short/long name) */
	/* FIXME: handle character set correctly (e.g. via iconv)
	 * c.f. EN 300 468 annex A */
	for (src = dest = s->service_name; *src; src++)
		if (*src >= 0x20 && (*src < 0x80 || *src > 0x9f))
			*dest++ = *src;
	*dest = '\0';
	if (!s->service_name[0]) {
		/* zap zero length names */
		free (s->service_name);
		s->service_name = 0;
	}

	MSG("0x%04x 0x%04x: pmt_pid 0x%04x %s -- %s (%s, %sscrambled)",
	    s->transport_stream_id,
	    s->service_id,
	    s->pmt_pid,
	    s->provider_name, s->service_name,
	    s->running == RM_NOT_RUNNING ? "not running" :
	    s->running == RM_STARTS_SOON ? "starts soon" :
	    s->running == RM_PAUSING     ? "pausing" :
	    s->running == RM_RUNNING     ? "running" : "???",
	    s->scrambled ? "" : "not ");
}

static int
find_descriptor(uint8_t tag, const unsigned char *buf,
		int descriptors_loop_len,
		const unsigned char **desc, int *desc_len)
{
	while (descriptors_loop_len > 0) {
		unsigned char descriptor_tag = buf[0];
		unsigned char descriptor_len = buf[1] + 2;

		if (!descriptor_len) {
			WARN("descriptor_tag == 0x%02x, descriptor_len == %i",
			     descriptor_tag, descriptor_len);
			break;
		}

		if (tag == descriptor_tag) {
			if (desc)
				*desc = buf;
			if (desc_len)
				*desc_len = descriptor_len;
			return 1;
		}

		buf += descriptor_len;
		descriptors_loop_len -= descriptor_len;
	}
	return 0;
}

static
void parse_descriptors (enum table_type t, const unsigned char *buf,
		int descriptors_loop_len, void *data)
{
	while (descriptors_loop_len > 0) {
		unsigned char descriptor_tag = buf[0];
		unsigned char descriptor_len = buf[1] + 2;

		if (!descriptor_len) {
			WARN("descriptor_tag == 0x%02x, descriptor_len == %i",
			     descriptor_tag, descriptor_len);
			break;
		}

		switch (descriptor_tag) {
		case 0x0a:
			if (t == PMT)
				parse_iso639_language_descriptor (buf, data);
			break;

		case 0x40:
			if (t == NIT)
				parse_network_name_descriptor (buf, data);
			break;

		case 0x43:
			if (t == NIT)
				parse_satellite_delivery_system_descriptor (buf, data);
			break;

		case 0x44:
			if (t == NIT)
				parse_cable_delivery_system_descriptor (buf, data);
			break;

		case 0x48:
			if (t == SDT)
				parse_service_descriptor (buf, data);
			break;

		case 0x5a:
			if (t == NIT)
				parse_terrestrial_delivery_system_descriptor (buf, data);
			break;

		case 0x62:
			if (t == NIT)
				parse_frequency_list_descriptor (buf, data);
			break;

		default:
			/*WARN("skip descriptor 0x%02x", descriptor_tag)*/;
		};

		buf += descriptor_len;
		descriptors_loop_len -= descriptor_len;
	}
}


static
void parse_pat (const unsigned char *buf, int section_length,
		int transport_stream_id)
{
	while (section_length > 0) {
		struct service *s;
		int service_id = (buf[0] << 8) | buf[1];

		if (service_id == 0) {
			buf += 4;		/*  skip nit pid entry... */
			section_length -= 4;
			continue;
		}
		s = find_or_alloc_entry_by_id (&service_list,
					       transport_stream_id,
					       service_id);

		s->pmt_pid = ((buf[2] & 0x1f) << 8) | buf[3];
		if (!s->priv && s->pmt_pid) {
			s->priv = malloc(sizeof(struct section_buf));
			setup_filter(s->priv, demux_devname,
				     s->pmt_pid, 0x02,
				     -1, -1, 1, 5);

			add_filter (s->priv);
		}

		buf += 4;
		section_length -= 4;
	};
}


static
void parse_pmt (const unsigned char *buf, int section_length, int service_id)
{
	int program_info_len;
	struct service *s;
        char msg_buf[14 * AUDIO_CHAN_MAX + 1];
        char *tmp;
        int i;

	s = find_or_alloc_entry_by_id (&service_list,
				       -1,
				       service_id);

	s->pcr_pid = ((buf[0] & 0x1f) << 8) | buf[1];

	program_info_len = ((buf[2] & 0x0f) << 8) | buf[3];

	buf += program_info_len + 4;
	section_length -= program_info_len + 4;

	while (section_length > 0) {
		int ES_info_len = ((buf[3] & 0x0f) << 8) | buf[4];
		int elementary_pid = ((buf[1] & 0x1f) << 8) | buf[2];

		switch (buf[0]) {
		case 0x01:
		case 0x02:
			if (s->video_pid == 0)
				s->video_pid = elementary_pid;
			break;
		case 0x03:
		case 0x04:
			if (s->audio_num < AUDIO_CHAN_MAX) {
				s->audio_pid[s->audio_num] = elementary_pid;
				parse_descriptors (PMT, buf + 5, ES_info_len, s);
				s->audio_num++;
			}
			else
				WARN("more than %i audio channels, truncating\n",
				     AUDIO_CHAN_MAX);
			break;
		case 0x06:
			if (find_descriptor(0x56, buf + 5, ES_info_len, NULL, NULL)) {
				MSG("  TELETEXT: PID 0x%04x", elementary_pid);
				s->teletext_pid = elementary_pid;
			}
			else if (find_descriptor(0x59, buf + 5, ES_info_len, NULL, NULL)) {
				/* Note: The subtitling descriptor can also signal
				 * teletext subtitling, but then the teletext descriptor
				 * will also be present; so we can be quite confident
				 * that we catch DVB subtitling streams only here, w/o
				 * parsing the descriptor. */
				MSG("  SUBTITLING: PID 0x%04x", elementary_pid);
				s->subtitling_pid = elementary_pid;
			}
			else if (find_descriptor(0x6a, buf + 5, ES_info_len, NULL, NULL)) {
				MSG("  AC3: PID 0x%04x", elementary_pid);
				s->ac3_pid = elementary_pid;
			}
			break;
		default:
			;
		};

		buf += ES_info_len + 5;
		section_length -= ES_info_len + 5;
	};


        tmp = msg_buf;
        tmp += sprintf(tmp, "0x%04x (%.4s)", s->audio_pid[0], s->audio_lang[0]);

	if (s->audio_num > AUDIO_CHAN_MAX) {
		WARN("more than %i audio channels: %i, truncating to %i\n",
		      AUDIO_CHAN_MAX, s->audio_num, AUDIO_CHAN_MAX);
		s->audio_num = AUDIO_CHAN_MAX;
	}

        for (i=1; i<s->audio_num; i++)
                tmp += sprintf(tmp, ", 0x%04x (%.4s)", s->audio_pid[i], s->audio_lang[i]);

        MSG("0x%04x 0x%04x: %s -- %s, pmt_pid 0x%04x, vpid 0x%04x, apid %s",
	    s->transport_stream_id,
	    s->service_id,
	    s->provider_name, s->service_name,
	    s->pmt_pid, s->video_pid, msg_buf);
}


static
void parse_nit (const unsigned char *buf, int section_length, int network_id)
{
	int descriptors_loop_len = ((buf[0] & 0x0f) << 8) | buf[1];
	struct network *n;

	if (section_length < descriptors_loop_len + 4)
	{
		WARN("network_id == 0x%04x, section_length == %i, "
		     "descriptors_loop_len == %i",
		     network_id, section_length, descriptors_loop_len);
		return;
	}

	n = find_or_alloc_entry_by_id (&network_list, 0, network_id);

	parse_descriptors (NIT, buf + 2, descriptors_loop_len, n);

	section_length -= descriptors_loop_len + 4;
	buf += descriptors_loop_len + 4;

	while (section_length > 6) {
		int transport_stream_id = (buf[0] << 8) | buf[1];
		struct transponder *t;

		descriptors_loop_len = ((buf[4] & 0x0f) << 8) | buf[5];

		if (section_length < descriptors_loop_len + 4 ||
		    !transport_stream_id)
		{
			WARN("transport_stream_id == 0x%04x, "
			     "section_length == %i, descriptors_loop_len == %i",
			     transport_stream_id, section_length,
			     descriptors_loop_len);
			break;
		}

		MSG("transport_stream_id 0x%04x", transport_stream_id);
		t = find_or_alloc_entry_by_id (&transponder_list,
					       network_id,
					       transport_stream_id);

		parse_descriptors (NIT, buf + 6, descriptors_loop_len, t);

		section_length -= descriptors_loop_len + 6;
		buf += descriptors_loop_len + 6;
	};
}


static
void parse_sdt (const unsigned char *buf, int section_length,
		int transport_stream_id)
{
	buf += 3;	       /*  skip original network id + reserved field */

	while (section_length > 4) {
		int service_id = (buf[0] << 8) | buf[1];
		int descriptors_loop_len = ((buf[3] & 0x0f) << 8) | buf[4];
		struct service *s;

		if (section_length < descriptors_loop_len ||
		    !transport_stream_id ||
		    !descriptors_loop_len)
		{
			WARN("service_id == 0x%02x, section_length == %i"
			     "descriptors_loop_len == %i",
			     service_id, section_length,
			     descriptors_loop_len);
			break;
		}

		s = find_or_alloc_entry_by_id (&service_list,
					       transport_stream_id,
					       service_id);

		s->running = (buf[3] >> 5) & 0x7;
		s->scrambled = (buf[3] >> 4) & 1;

		parse_descriptors (SDT, buf + 5, descriptors_loop_len, s);

		section_length -= descriptors_loop_len + 5;
		buf += descriptors_loop_len + 5;
	};
}


static
int get_bit (uint8_t *bitfield, int bit)
{
	return (bitfield[bit/8] >> (bit % 8)) & 1;
}

static
void set_bit (uint8_t *bitfield, int bit)
{
	bitfield[bit/8] |= 1 << (bit % 8);
}


/**
 *   returns 0 when more sections are expected
 *	   1 when all sections are read on this pid
 *	   -1 on invalid table id
 */
static
int parse_section (struct section_buf *s)
{
	const unsigned char *buf = s->buf;
	int table_id;
	int section_length;
	int id;
	int section_version_number;
	int section_number;
	int last_section_number;
	int i, j;

	table_id = buf[0];

	for (i=0; i<3; i++)
		if (s->table_id[i] == table_id)
			break;

	if (i > 2)
		return -1;

	section_length = (((buf[1] & 0x0f) << 8) | buf[2]) - 11;

	id = (buf[3] << 8) | buf[4];
	section_version_number = (buf[5] >> 1) & 0x1f;
	section_number = buf[6];
	last_section_number = buf[7];

	if (s->section_version_number[i] != section_version_number ||
	    s->id[i] != id)
	{
		if (s->section_version_number[i] != -1 && s->id[i] != -1)
			WARN("section_version_number or id changed %d -> %d / %04x -> %04x",
				s->section_version_number[i], section_version_number,
				s->id[i], id);
		s->id[i] = id;
		s->section_version_number[i] = section_version_number;
		s->sectionfilter_done[i] = 0;
		memset (s->section_done[i], 0, sizeof(s->section_done[i]));
	}

	buf += 8;

	if (!get_bit(s->section_done[i], section_number)) {
		set_bit (s->section_done[i], section_number);

		MSG("pid 0x%02x tid 0x%02x id 0x%04x, "
		    "%i/%i (version %i)",
		    s->pid, table_id, id, section_number,
		    last_section_number, section_version_number);

		switch (table_id) {
		case 0x00:
			parse_pat (buf, section_length, id);
			break;

		case 0x02:
			parse_pmt (buf, section_length, id);
			break;

		case 0x40:
		case 0x41:
			MSG("NIT (%s TS)\n", table_id == 0x40 ? "actual":"other");
			parse_nit (buf, section_length, id);
			break;

		case 0x42:
		case 0x46:
			MSG("SDT (%s TS)\n", table_id == 0x42 ? "actual":"other");
			parse_sdt (buf, section_length, id);
			break;

		default:
			;
		};

		for (j=0; j<=last_section_number; j++)
			if (get_bit (s->section_done[i], j) == 0)
				break;

		if (j > last_section_number)
			s->sectionfilter_done[i] = 1;
	}

	if (s->sectionfilter_done[0] &&
	    s->sectionfilter_done[1] &&
	    s->sectionfilter_done[2])
		return 1;

	return 0;
}


static
int read_sections (struct section_buf *s)
{
	int section_length, count;

	if (s->sectionfilter_done[0] &&
	    s->sectionfilter_done[1] &&
	    s->sectionfilter_done[2])
		return 1;

	/* the section filter API guarantess that we get one full section
	 * per read(), provided that the buffer is large enough (it is)
	 */
	if (((count = read (s->fd, s->buf, sizeof(s->buf))) < 0) && errno == EOVERFLOW)
		count = read (s->fd, s->buf, sizeof(s->buf));
	if (count < 0) {
		MSG("read_sections: read error %d %m\n", errno);
		return -1;
	}

	if (count < 4)
		return -1;

	section_length = ((s->buf[1] & 0x0f) << 8) | s->buf[2];

	if (count != section_length + 3)
		return -1;

	if (parse_section(s) == 1)
		return 1;

	return 0;
}


static LIST_HEAD(running_filters);
static LIST_HEAD(waiting_filters);
static int n_running;
#define MAX_RUNNING 32
static struct pollfd poll_fds[MAX_RUNNING];
static struct section_buf* poll_section_bufs[MAX_RUNNING];


static
void setup_filter (struct section_buf* s, const char *dmx_devname,
		   int pid, int tid0, int tid1, int tid2,
		   int run_once, int timeout)
{
	int i;

	memset (s, 0, sizeof(struct section_buf));

	s->fd = -1;
	s->dmx_devname = dmx_devname;
	s->pid = pid;
	s->table_id[0] = tid0;
	s->table_id[1] = tid1;
	s->table_id[2] = tid2;

	s->run_once = run_once;

	if (long_timeout)
		s->timeout = 5 * timeout;
	else
		s->timeout = timeout;

	for (i=0; i<3; i++) {
		s->id[i] = -1;
		s->section_version_number[i] = -1;
	}

	memset (s->section_done, 0, sizeof(s->section_done));

	INIT_LIST_HEAD (&s->list_head);
}

static void update_poll_fds(void)
{
	struct list_head *p, *n;
	struct section_buf* s;
	int i;

	memset(poll_section_bufs, 0, sizeof(poll_section_bufs));
	for (i = 0; i < MAX_RUNNING; i++)
		poll_fds[i].fd = -1;
	i = 0;
	list_for_each_safe (p, n, &running_filters) {
		if (i >= MAX_RUNNING)
			PERROR("fatal: too many poll_fds");
		s = list_entry (p, struct section_buf, list_head);
		if (s->fd == -1)
			PERROR("fatal: s->fd == -1 on running_filters");
		poll_fds[i].fd = s->fd;
		poll_fds[i].events = POLLIN;
		poll_fds[i].revents = 0;
		poll_section_bufs[i] = s;
		i++;
	}
	if (i != n_running)
		PERROR("fatal: n_running is hosed");
}

static
int start_filter (struct section_buf* s)
{
	struct dmx_sct_filter_params f;
	int i;

	if (n_running >= MAX_RUNNING)
		goto err0;
	if ((s->fd = open (s->dmx_devname, O_RDWR | O_NONBLOCK)) < 0)
		goto err0;

	//MSG("start filter pid 0x%04x", s->pid);

	memset(&f, 0, sizeof(struct dmx_sct_filter_params));

	f.pid = (uint16_t) s->pid;

	if (s->table_id[1] == -1 && s->table_id[2] == -1 &&
	    s->table_id[0] < 0x100 && s->table_id[0] > 0)
	{
		f.filter.filter[0] = (uint8_t) s->table_id[0];
		f.filter.mask[0]   = 0xff;
	}

	for (i=0; i<3; i++)
		if (s->table_id[i] == -1)
			s->sectionfilter_done[i] = 1;
		else
			s->sectionfilter_done[i] = 0;

	f.timeout = 0;
	f.flags = DMX_IMMEDIATE_START | DMX_CHECK_CRC;

	if (ioctl(s->fd, DMX_SET_FILTER, &f) == -1) {
		PERROR ("ioctl DMX_SET_FILTER failed");
		goto err1;
	}

	time (&s->start_time);

	list_del_init (&s->list_head);  /* might be in waiting filter list */
	list_add (&s->list_head, &running_filters);

	n_running++;
	update_poll_fds();

	return 0;

err1:
	ioctl (s->fd, DMX_STOP);
	close (s->fd);
err0:
	return -1;
}


static
void stop_filter (struct section_buf *s)
{
	//MSG("stop filter pid 0x%04x", s->pid);
	ioctl (s->fd, DMX_STOP);
	close (s->fd);
	s->fd = -1;
	list_del (&s->list_head);
	s->running_time += time(NULL) - s->start_time;

	n_running--;
	update_poll_fds();
}


void add_filter (struct section_buf *s)
{
	MSG("add filter pid 0x%04x", s->pid);
	if (start_filter (s))
		list_add_tail (&s->list_head, &waiting_filters);
}


void remove_filter (struct section_buf *s)
{
	MSG("remove filter pid 0x%04x", s->pid);
	stop_filter (s);

	while (!list_empty(&waiting_filters)) {
		struct list_head *next = waiting_filters.next;
		s = list_entry (next, struct section_buf, list_head);
		if (start_filter (s))
			break;
	};
}


void reschedule_filter (struct section_buf *s)
{
	remove_filter (s);
	add_filter (s);
}


void read_filters (void)
{
	struct section_buf *s;
	int i, n, done;

	n = poll(poll_fds, n_running, 1000);
	if (n == -1)
		PERROR("poll");

	for (i = 0; i < n_running; i++) {
		s = poll_section_bufs[i];
		if (!s)
			PERROR("fatal: poll_section_bufs[%d] is NULL", i);
		if (poll_fds[i].revents)
			done = read_sections (s) == 1;
		else
			done = 0; /* timeout */
		if (done || time(NULL) > s->start_time + s->timeout) {
			if (s->run_once) {
				if (done)
					MSG("filter done pid 0x%04x", s->pid);
				else
					MSG("filter timeout pid 0x%04x", s->pid);
				remove_filter (s);
			}
		}
	}
}


static
int mem_is_zero (const void *mem, int size)
{
	const char *p = mem;

	while (size) {
		if (*p != 0x00)
			return 0;
		p++;
	}

	return 1;
}


#define SWITCHFREQ 11700000
#define LOF_HI     10600000
#define LOF_LO      9750000

static int switch_pos = 0;

static
int __tune_to_transponder (int frontend_fd, struct transponder *t)
{
	struct dvb_frontend_parameters p;
	int i;

	if (mem_is_zero (&t->param, sizeof(struct dvb_frontend_parameters)))
		return -1;

	memcpy (&p, &t->param, sizeof(struct dvb_frontend_parameters));

	MSG(">>> tune to: ");
	dump_dvb_parameters (stderr, t->type, &p);
	if (t->scan_done)
		fprintf (stderr, " (done)");
	if (t->last_tuning_failed)
		fprintf (stderr, " (tuning failed)");
	MSG("\n");

	if (t->type == FE_QPSK) {
		int hiband = (p.frequency >= SWITCHFREQ);

		setup_switch (frontend_fd,
			      switch_pos,
			      t->polarisation == POLARISATION_VERTICAL ? 0 : 1,
			      hiband);
		if (hiband)
			p.frequency -= LOF_HI;
		else
			p.frequency -= LOF_LO;
	}

	ioctl(frontend_fd, FE_SET_FRONTEND, &p);

	for (i=0; i<3; i++) {
		fe_status_t s;

		ioctl(frontend_fd, FE_READ_STATUS, &s);

		MSG(">>> tuning status == 0x%02x!!!", s);

		if (s & FE_HAS_LOCK) {
			t->last_tuning_failed = 0;
			return 0;
		}

		usleep (500000);
	}

	MSG(">>> tuning failed!!!\n");

	t->last_tuning_failed = 1;

	return -1;
}

static
int tune_to_transponder (int frontend_fd, struct transponder *t)
{
	if (__tune_to_transponder (frontend_fd, t) == 0)
		return 0;

	return __tune_to_transponder (frontend_fd, t);
}


#include "initial.h"

static
int tune_initial (int frontend_fd)
{
	struct dvb_frontend_info fe_info;
	struct transponder *initial_list;
	int list_size;
	static int index = 0;

	ioctl(frontend_fd, FE_GET_INFO, &fe_info);

	switch (fe_info.type) {
	case FE_QPSK:
		initial_list = qpsk_probes;
		list_size = sizeof(qpsk_probes)/sizeof(struct transponder);
		break;
	case FE_QAM:
		initial_list = qam_probes;
		list_size = sizeof(qam_probes)/sizeof(struct transponder);
		break;
	case FE_OFDM:
		initial_list = ofdm_probes;
		list_size = sizeof(ofdm_probes)/sizeof(struct transponder);
		break;
	default:
		WARN("FE_GET_INFO returned unknown frontend type!");
		return -1;
	};

	while (index < list_size) {
		struct transponder *t = &initial_list[index];

		if (tune_to_transponder (frontend_fd, t) == 0)
			return 0;

		index++;
	}

	index = 0;

	return -1;
}


static
int tune_to_next_transponder (int frontend_fd)
{
	struct transponder *t;
	int i;

	for (i = 0; ; i++) {
		if (!(t = find_nth_entry (&transponder_list, i)))
			break;
		if (t->scan_done || t->last_tuning_failed)
			continue;
		t->scan_done = 1;
retry:
		if (tune_to_transponder (frontend_fd, t) == 0)
			return 0;
		if (t->other_frequency_flag &&
				t->other_f &&
				t->n_other_f) {
			t->param.frequency = t->other_f[t->n_other_f - 1];
			t->n_other_f--;
			MSG("retrying with f=%d", t->param.frequency);
			goto retry;
		}
	}
	return -1;
}



static
void scan_tp (void)
{
	struct section_buf s0;
	struct section_buf s1;
	struct section_buf s2;

	/**
	 *  filter timeouts > min repetition rates specified in ETR211
	 */
	setup_filter (&s0, demux_devname, 0x00, 0x00, -1, -1, 1, 5); /* PAT */
	setup_filter (&s1, demux_devname, 0x11, 0x42, -1, -1, 1, 5); /* SDT */

	add_filter (&s0);
	add_filter (&s1);

	if (!current_tp_only || output_format != OUTPUT_PIDS) {
		setup_filter (&s2, demux_devname, 0x10, 0x40, -1, -1, 1, 15); /* NIT */
		add_filter (&s2);
	}

	do {
		read_filters ();
	} while (!(list_empty(&running_filters) &&
		   list_empty(&waiting_filters)));
}

static
void scan_network (const char *frontend_devname)
{
	int frontend_fd;

	MSG ("using '%s' and '%s'\n", frontend_devname, demux_devname);

	if ((frontend_fd = open (frontend_devname, O_RDWR)) < 0)
		PERROR("failed opening '%s'", frontend_devname);

	if (tune_initial (frontend_fd) < 0) {
		MSG("initial tuning failed!");
		return;
	}

	do {
		scan_tp();
	} while (tune_to_next_transponder(frontend_fd) == 0);

	close (frontend_fd);
}


static void pids_dump_service_parameter_set(FILE *f, struct service *s)
{
        int i;

	fprintf(f, "%-24.24s (0x%04x) %02x: ", s->service_name, s->service_id, s->type);
	if (!s->pcr_pid || (s->type > 2))
		fprintf(f, "           ");
	else if (s->pcr_pid == s->video_pid)
		fprintf(f, "PCR == V   ");
	else if ((s->audio_num == 1) && (s->pcr_pid == s->audio_pid[0]))
		fprintf(f, "PCR == A   ");
	else
		fprintf(f, "PCR 0x%04x ", s->pcr_pid);
	if (s->video_pid)
		fprintf(f, "V 0x%04x", s->video_pid);
	else
		fprintf(f, "        ");
	if (s->audio_num)
		fprintf(f, " A");
        for (i = 0; i < s->audio_num; i++) {
		fprintf(f, " 0x%04x", s->audio_pid[i]);
		if (s->audio_lang[i][0])
			fprintf(f, " (%.3s)", s->audio_lang[i]);
		else if (s->audio_num == 1)
			fprintf(f, "      ");
	}
	if (s->teletext_pid)
		fprintf(f, " TT 0x%04x", s->teletext_pid);
	if (s->ac3_pid)
		fprintf(f, " AC3 0x%04x", s->ac3_pid);
	if (s->subtitling_pid)
		fprintf(f, " SUB 0x%04x", s->subtitling_pid);
	fprintf(f, "\n");
}

static
void dump_lists (void)
{
	int i;

	MSG("dumping lists (%d services)", service_list.count);
	for (i=0; i<service_list.count; i++) {
		struct service *s = find_nth_entry (&service_list, i);
		struct transponder *t;
		//struct network *n;
		char sn[20];

		t = find_entry_by_id (&transponder_list, -1,
				      s->transport_stream_id);

		if (t || output_format == OUTPUT_PIDS) {
			//n = find_entry_by_id (&network_list, -1,
			//		      t->network_id);

			if (!s->service_name) {
				/* not in SDT */
				snprintf(sn, sizeof(sn), "[%04x]", s->service_id);
				s->service_name = strdup(sn);
			}
                        switch (output_format)
                        {
                          case OUTPUT_PIDS:
                                pids_dump_service_parameter_set (stdout, s);
                                break;
                          case OUTPUT_VDR:
                                vdr_dump_service_parameter_set (stdout,
						    s->service_name,
						    t->type,
						    &t->param,
						    s->video_pid,
						    s->audio_pid,
						    //FIXME: s->audio_lang
						    //FIXME: s->teletext_pid
						    //FIXME: s->subtitling_pid
						    //FIXME: s->ac3_pid
                                                    s->audio_num,
                                                    s->service_id);
                                break;
                          case OUTPUT_ZAP:
			        zap_dump_service_parameter_set (stdout,
						    s->service_name,
						    t->type,
						    &t->param,
						    s->video_pid,
						    s->audio_pid);
                          default:
                                break;
                          }
		}
		else
			MSG("transport_stream_id 0x%04x not found", s->transport_stream_id);
	}
	MSG("Done.");
}


static const char *usage =
"\nusage: %s [-c] [-o output_format] [-a adapter_num] [-f frontend_id] [-d demux_id] [-s switch_pos]\n"
"\t-c\tscan on currently tuned transponder only\n"
"\t-5\tmultiply all filter timeouts by factor 5\n"
"\t\tfor non-DVB-compliant section repitition rates\n"
"\t-o\tSelect output format: 'zap' (default) or 'vdr' or 'pids' (default with -c)\n";


int main (int argc, char **argv)
{
	char frontend_devname [80];
	int adapter = 0, frontend = 0, demux = 0;
	int opt, i;

	while ((opt = getopt(argc, argv, "5ca:f:d:s:o:")) != -1) {
		switch (opt) {
		case 'a':
			adapter = strtoul(optarg, NULL, 0);
			break;
		case 'c':
			current_tp_only = 1;
			output_format = OUTPUT_PIDS;
			break;
		case 'd':
			demux = strtoul(optarg, NULL, 0);
			break;
		case 'f':
			frontend = strtoul(optarg, NULL, 0);
			break;
		case 's':
			switch_pos = strtoul(optarg, NULL, 0);
			break;
		case 'o':
                        if      (strcmp(optarg, "zap") == 0) output_format = OUTPUT_ZAP;
                        else if (strcmp(optarg, "vdr") == 0) output_format = OUTPUT_VDR;
                        else if (strcmp(optarg, "pids") == 0) output_format = OUTPUT_PIDS;
                        else {
				fprintf (stderr, usage, argv[0]);
				return -1;
			}
			break;
		case '5':
			long_timeout = 1;
			break;
		default:
			fprintf (stderr, usage, argv[0]);
			return -1;
		};
	}

	if (switch_pos >= 4) {
		fprintf (stderr, "switch position needs to be < 4!\n");
		return -1;
	}

	snprintf (frontend_devname, sizeof(frontend_devname),
		  "/dev/dvb/adapter%i/frontend%i", adapter, frontend);

	snprintf (demux_devname, sizeof(demux_devname),
		  "/dev/dvb/adapter%i/demux%i", adapter, demux);

	for (i = 0; i < MAX_RUNNING; i++)
		poll_fds[i].fd = -1;

	if (current_tp_only)
		scan_tp ();
	else
		scan_network (frontend_devname);

	dump_lists ();

	return 0;
}

static void dump_dvb_parameters (FILE *f, fe_type_t type, struct dvb_frontend_parameters *p)
{
  switch (output_format) {
    case OUTPUT_PIDS:
    case OUTPUT_VDR:
        vdr_dump_dvb_parameters (f, type, p);
        break;
    case OUTPUT_ZAP:
        zap_dump_dvb_parameters (f, type, p);
        break;
    default:
        break;
  }
}
