/* 
    ntuxplayer

    Copyright (C) 2000 Marcus Metzler (mocm@convergence.de)
    for convergence integrated media

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <libgen.h>
#include <stdint.h>

#include <getopt.h>
#include <cdk/cdk.h>

#include "dvb_formats.h"

#define IPACKS 2048
#define MAXAPIDS 32
int yskip = 0;
int loop=0;

#if 1
#define DVR_DEV   "/dev/dvb/adapter%d/dvr0"	
#define VIDEO_DEV "/dev/dvb/adapter%d/video0"
#define AUDIO_DEV "/dev/dvb/adapter%d/audio0"
#define DEMUX_DEV "/dev/dvb/adapter%d/demux0"
#else
#define DVR_DEV   "/dev/ost/dvr%d"
#define VIDEO_DEV "/dev/ost/video%d"
#define AUDIO_DEV "/dev/ost/audio%d"
#define DEMUX_DEV "/dev/ost/demux%d"
#endif

static inline ssize_t my_read(int fd, void *buf, size_t count)
{
	return read(fd, buf, count);
}

static ssize_t my_write(int fd, uint8_t *buf, int count)
{
	return write(fd, buf, count);
}

void stop_curses(int dummy,void *dummy2)
{
    clear();
    refresh();
    endwin();
}

void curses_init(void)     /* for kbd input */
{
    initscr();
    refresh();
    cbreak();
    noecho();
    nonl();
    intrflush(stdscr, FALSE);
    keypad(stdscr, TRUE);
    nodelay(stdscr, TRUE);

    on_exit(stop_curses, NULL);
}

static void sw_mess(int cc, av_settings *av){
	switch(cc){
	case 'z':
		mvprintw(yskip+1,30,
			 "playback frozen   ");
		break;
	case 's':
		mvprintw(yskip+1,30,
			 "playback stopped  ");
		break;
		
	case 'c':
		mvprintw(yskip+1,30,
			 "playback continued ");
		break;
		
	case 'p':
		mvprintw(yskip+1,30,
			 "playback started  ");
		break;
		
	case 'f':
		mvprintw(yskip+1,30,
			 "fast forward      ");
		break;
		
	case 'l':
		loop = !loop;
		if (loop)
			mvprintw(yskip+1,30,
				 "loop on           ");
		else 
			mvprintw(yskip+1,30,
				 "loop off          ");
		break;
		
	case 'm':
		mvprintw(yskip+1,30,
			 "slow motion       ");
		break;
		
	case '+':
		mvprintw(yskip+2,30,
			 "Volume: %d",av->volume);
		break;
		
	case '-':
		mvprintw(yskip+2,30,
			 "Volume: %d",av->volume);
		break;
		
	case 258:
		mvprintw(yskip+1,30,
			 "big skip back          ");
		break;
		
	case 260:
		mvprintw(yskip+1,30,
			 "skip back              ");
		break;
		
	case 259:
		mvprintw(yskip+1,30,
			 "big skip ahead         ");
		break;
		
	case 261:
		mvprintw(yskip+1,30,
			 "skip ahead              ");
		break;
	}
}

void play_file_video_stdin(int fd, int fda, av_settings *av)
{
	switch_t sw;

	sw.fdv = fd;
	sw.fda = fda;
	sw.filefd = STDIN_FILENO;
	sw.av = av;
	sw.ts = 0;
	sw.stop = NULL;

	play_file_video_mpg(&sw);
}
	

int play_file_video(int filefd, int fd, int fda, int next, av_settings *av)
{
	char buf[BUFFY];
	int count,c;
	int written;
	struct pollfd pfd[NFD];
	uint64_t length = 0;
	ipack pack;
	uint64_t l = 0;
	switch_t sw;

	sw.stop = NULL;

	pfd[0].fd = STDIN_FILENO;
	pfd[0].events = POLLIN;
	
	pfd[1].fd = fd;
	pfd[1].events = POLLOUT;

	if (fda) audioSetAVSync(fda, true);
	if (!next){
		videoSelectSource(fd,VIDEO_SOURCE_MEMORY);
		videoPlay(fd);
		audioPlay(fda);
	}		

	if (av->needs_demux){
		init_ipack(&pack, IPACKS, write_out_demux,0);
		pack.data = (void *)&pack;
		pack.fd   = fd;
		pack.fd1  = fd;
		pack.fd2  = fda;
		pack.av   = av;
	}

	length = lseek(filefd, 0, SEEK_END);
	lseek(filefd,0,SEEK_SET);
	mvprintw(yskip,0,"File length: %2.2f MB                      ",
		 (double)length/1024/1024);
	count = my_read(filefd,buf,BUFFY);
	my_write(fd,buf,count);
	if (count > 0)  l+= count;
	
	set_volume(fda, av->volume, av);
	mvprintw(yskip+2,30,
		 "Volume: %d",av->volume);
	mvprintw(yskip+1,30, "playback started  ");
	refresh();

	sw.fdv = fd;
	sw.fda = fda;
	sw.filefd = filefd;
	sw.av = av;
	sw.ts = 0;
	sw.stop = NULL;

	while ( l < length ){
		count = my_read(filefd,buf,BUFFY);

		if (count > 0) l+= count;
		mvprintw(yskip+2,0,"read %03.2f%%",100.0*l/length);
		refresh();

		written = 0;
		while(count >0 && written < count){
			if (av->needs_demux){
				instant_repack(buf, count, &pack);
				written = count;
#ifdef HAVE_LIBA52
				if (av->soft_ac3)
					mvprintw(yskip+4,0,"a52_track: 0x%02x",
						 av->a52_track);
#endif
			}
			if (poll(pfd,NFD,100)){
				if (!av->needs_demux && 
				    pfd[1].revents & POLLOUT){
					c = my_write(fd,buf+written,
							count-written);
					if (c > 0) written += c;
				}
				if (pfd[0].revents & POLLIN){
					int cc = getch();
					sw_mess(cc, av);
					cc = av_switches(cc,&sw,&l); 
					switch(cc){
						
					case 0:
					case 1:
						return cc;
					case 2:
						videoClear(fd);
						audioClear(fda);
						written = count;
#ifdef HAVE_LIBA52
						if (av->soft_ac3)
							init_a52_buf();
#endif
						break;

					case -1:
						break;
					}
				}
				
			}
		}
	}
	usleep(1000);
	if (fda) audioSetAVSync(fda, false);
	return 0;
}

void play_file_dvr_stdin(int fd_dvr, int fda, int fdv, uint16_t apid, uint16_t vpid, int dev, av_settings *av)
{
	switch_t sw;

	sw.fdv = fdv;
	sw.fda = fda;
	sw.filefd = STDIN_FILENO;
	sw.av = av;
	sw.ts = 1;
	sw.stop = NULL;

	play_file_dvr_ts(fd_dvr, apid, vpid, dev, &sw);
}


int play_file_dvr(int fd_dvr, int fda, int fdv, int filefd, 
		   uint16_t apid, uint16_t vpid, int dev, av_settings *av)
{
	char buf[BUFFY2];
	int count, c;
	int written;
	int stopped = 0;
	struct pollfd pfd[NFD2];
	uint16_t ppid=0;
	int fd_vdemux;
	int fd_ademux;
	int fd;
	uint64_t length = 0;
	uint64_t l = 0;
	int stop = 0;
	char devnamed[80];
	uint16_t apids[MAXAPIDS];
	int napid=0, a;
	switch_t sw;
		
	sw.stop = NULL;
	
	sprintf(devnamed,DEMUX_DEV,dev);

	if((fd = open(devnamed,O_RDWR|O_NONBLOCK)) < 0){
		perror(devnamed);
		return 0;
	}

	pfd[0].fd = STDIN_FILENO;
	pfd[0].events = POLLIN;
	
	if (av->nopat) find_avpids(filefd, &vpid, &apid);
	length = lseek(filefd, 0, SEEK_END);
	lseek(filefd,0,SEEK_SET);
	mvprintw(yskip-2,0,"File length: %2.2f MB               ",(double)length/(1024*1024));

	if (apid && vpid) ppid = 1;
	else {
		apid = 0;
		vpid = 0;
		ppid = 0;
	}


	if ((fd_vdemux=open(devnamed, O_RDWR|O_NONBLOCK)) < 0){
		perror(devnamed);
		return 0;
	}
	if ((fd_ademux=open(devnamed, O_RDWR|O_NONBLOCK)) < 0){
		perror(devnamed);
		return 0;
	}
				
	set_pat_filt(fd);

	if (vpid && apid) set_av_filts(fd_vdemux, fd_ademux, vpid, apid);
	mvprintw(yskip+2,0,
		 "apid %d (0x%02x)   ",apid,apid);
	mvprintw(yskip+3,0,
		 "vpid %d (0x%02x)   ",vpid,vpid);

	
	set_volume(fda, av->volume, av);
	mvprintw(yskip+1,30,
		 "Volume: %d",av->volume);

	mvprintw(yskip,30, "playback started  ");
	refresh();

	sw.fdv = fdv;
	sw.fda = fda;
	sw.filefd = filefd;
	sw.av = av;
	sw.ts = 1;


	while ( (count = my_read(filefd,buf,BUFFY2)) > 0  && !stop){
		if (count > 0) l+= count;
		if (apid && vpid){
			mvprintw(yskip,0,"read %03.2f%%    ",100.0*l/length);
			refresh();
		}
		written = 0;
		if(!ppid && !av->nopat){
			ppid = get_pmt_pid(fd);
			if (ppid) set_pmt_filt(fd,ppid);
		} else if (!vpid && !apid){
			if (!av->nopat){
				//napid = new_get_av_pids( fd, &vpid, apids, 
				//			 MAXAPIDS);
				//apid = apids[0];
				get_av_pids( fd, &vpid, &apid);
			}
			if (vpid && apid) 
				set_av_filts(fd_vdemux, fd_ademux, vpid, apid);
			mvprintw(yskip+2,0,
				 "apid %d (0x%02x)   ",apid,apid);
			mvprintw(yskip+3,0,
				 "vpid %d (0x%02x)   ",vpid,vpid);

			
			mvprintw(yskip+4,0," found apids: ");
			for ( a = 0; a < napid; a++){
				mvprintw(yskip+4,14+10*a,"%d: %d (0x%02x)   ",
					 a+1, apids[a], apids[a]);
			}
		}
		while( count > 0 && written < count){
			if (!stopped ){
				c = my_write(fd_dvr,buf+written, count-written);
				if (c>0) written += c;
			}
			if (poll(pfd,NFD2,100)){
				
				if (pfd[0].revents & POLLIN){
					int cc = getch();
					sw_mess(cc, av);
					cc = av_switches(cc,&sw,&l); 
					switch(cc){
						
					case 0:
					case 1:
						stop = cc+1;
						stop_av_filts(fd_ademux, 
							      fd_vdemux);
						break;
					case 2:
						videoClear(fdv);
						audioClear(fda);
						written = count;
						break;
						
					case 3:
						stopped = 0;
						break;

					case 4:
						stopped = 1;
						break;

					case -1:
						break;
					}
				}
			}
		}
	}

        close (fd_vdemux);
        close (fd_ademux);
        close (fd);

	if (stop) return stop-1;
	else return 0;
}


void write_out(uint8_t *buf, int count, void *priv)
{
	struct pollfd pfd[2];
	ipack *p = (ipack *) priv;
	uint64_t l;
	int written,c;
	int fd = p->fd;
	int filefd = p->ffd;
	switch_t sw;

	sw.stop = NULL;

	if (fd <0 ) return;

	pfd[1].fd = fd;
	pfd[1].events = POLLOUT;
	pfd[0].fd = STDIN_FILENO;
	pfd[0].events = POLLIN;

	
	set_volume(p->fd2, p->av->volume, p->av);
	mvprintw(yskip+2,30,"Volume: %d",p->av->volume);

	if(p->playing < 4 && p->fd == p->fd1){
		videoPlay(p->fd1);
		audioPlay(p->fd2);
		p->playing++;  
	}

	sw.fdv = p->fd1;
	sw.fda = p->fd2;
	sw.filefd = filefd;
	sw.av = p->av;
	sw.ts = 1;


	written = 0;
	while(written < count){
		if (poll(pfd,2,100)){
			if (pfd[1].revents & POLLOUT){
				if (p->av->needs_demux){
					c = write_demux(fd,buf+written,
						     count-written, p->av,
							my_write);
					if (c>0) written += c;
				} else {
					c = my_write(fd,buf+written,
						     count-written);
					if (c>0) written += c;
				}
			}
   			if (pfd[0].revents & POLLIN){

					int cc = getch();
					sw_mess(cc, p->av);
					cc = av_switches(cc,&sw,&l); 
					switch(cc){
						
					case 0:
						p->playing=0;
						break;
					case 1:
						p->playing=-1;
						break;
					case 2:
						videoClear(p->fd1);
						audioClear(p->fd2);
						written = count;
						break;
						
					case -1:
						break;
					}
			}
		}
	}
}

int play_ts_video(int filefd, int fd, int fda, uint16_t pida, 
		  uint16_t pidv, av_settings *av)
{
	uint8_t buf[IN_SIZE];
	uint8_t mbuf[TS_SIZE];
	int count;
	uint64_t length = 0;
	int64_t l = 0;
	int i;
	uint16_t pid;
	ipack pa, pv;
	ipack *p;

	videoSetBlank(fd,false);
	audioSelectSource(fda,AUDIO_SOURCE_MEMORY);
	videoSelectSource(fd,VIDEO_SOURCE_MEMORY);
	audioSetAVSync(fda, true);

	if (av->nopat) 
		find_avpids(filefd, &pidv, &pida);

	init_ipack(&pa, IPACKS, write_out,0);
	init_ipack(&pv, IPACKS, write_out,0);
	length = lseek(filefd, 0, SEEK_END);
	lseek(filefd,0,SEEK_SET);
	mvprintw(yskip-1,0,"File length: %2.2f MB           ",
		 (double)length/(1024*1024));
	refresh();

	pa.data = (void *)&pa;
	pv.data = (void *)&pv;
	pa.av = av;
	pv.av = av;
	pv.fd   = fd;
	if (av->needs_demux)
		pa.fd   = fda;
	else 
		pa.fd   = fd;
		
	pv.fd1  = fd;
	pa.fd1  = fd;
	pv.fd2  = fda;
	pa.fd2  = fda;
	pv.ffd  = filefd;
	pa.ffd  = filefd;

	count = my_read(filefd,mbuf,TS_SIZE);
	l+= count;
	for ( i = 0; i < 188 ; i++){
		if ( mbuf[i] == 0x47 ) break;
	}


	if ( i == 188){
		mvprintw(yskip+1,0,"Not a TS\n");
		yskip++;
		refresh();
	} else {
		memcpy(buf,mbuf+i,TS_SIZE-i);
		count = my_read(filefd,mbuf,i);
		l+= count;
		memcpy(buf+TS_SIZE-i,mbuf,i);
		i = 188;
	}

	mvprintw(yskip+3,0,
		 "apid %d (0x%02x)   ",pida,pida);
	mvprintw(yskip+4,0,
		 "vpid %d (0x%02x)   ",pidv,pidv);

	pv.playing = 1;
	pa.playing = 1;
	mvprintw(yskip+1,30, "playback started  ");

	while ( l < length && pv.playing > 0 && pa.playing > 0){
		if ( ( count = my_read(filefd,buf+i,IN_SIZE-i)+i) < 0){
		        perror("ts play ");
			exit(1);
		}		

		if (count < 1) break;
		
		l = lseek(filefd, 0, SEEK_CUR);
		mvprintw(yskip+1,0,"read %03.2f%%              ",100.0*l/length);
		refresh();

		for( i = 0; i < count; i+= TS_SIZE){
			uint8_t off = 0;

			if ( count - i < TS_SIZE) break;

			pid = get_pid(buf+i+1);
			if (!(buf[3+i]&0x10)) // no payload?
				continue;
			if (pid == pidv){
				p = &pv;
			} else {
				if (pid == pida){
					p = &pa;
				} else continue;
			}
			if ( buf[1+i]&0x40) {
				if (p->plength == MMAX_PLENGTH-6){
					p->plength = p->found-6;
					p->found = 0;
					send_ipack(p);
					reset_ipack(p);
				}
			}

			if ( buf[3+i] & 0x20) {  // adaptation field?
				off = buf[4+i] + 1;
			}
			instant_repack(buf+4+off+i, TS_SIZE-4-off, p);
		}
		i = 0;

	}

	audioSelectSource(fda,AUDIO_SOURCE_DEMUX);
	videoSelectSource(fd,VIDEO_SOURCE_DEMUX);
	videoPlay(fd);
	audioSetAVSync(fda, true);
	if ( pv.playing < 0 || pa.playing < 0) return 1;
	else return 0;

}


void play_ts_video_stdin(int fdv, int fda, int apid, int vpid, 
			 av_settings *av)
{
	switch_t sw;

	sw.fdv = fdv;
	sw.fda = fda;
	sw.filefd = STDIN_FILENO;
	sw.av = av;
	sw.ts = 1;
	sw.stop = NULL;

	play_ts_video_n(&sw, apid, vpid);
}


void usage(char *progname)
{
	fprintf(stderr,"usage: %s [options] <input files>\n\n",progname);
	fprintf(stderr,"options:\n");
	fprintf(stderr,"  --help,             -h:  print help message\n");  
	fprintf(stderr,"  --input_stream,     -i:  set input stream type (PS (default), TS)\n");
	fprintf(stderr,"  --audio_out,        -e:  set audio stream type (MP2(default), AC3 and others)\n");
	fprintf(stderr,"  --audio_pid,        -a:  audio PID for TS (0 = auto)\n");
	fprintf(stderr,"  --video_pid,        -v:  audio PID for TS (0 = auto)\n");
	fprintf(stderr,"  --device,           -d:  device nb.(0 = default)\n");
	fprintf(stderr,"  --no_dvr,           -n:  do not use dvr device\n");
	fprintf(stderr,"  --stdin,            -o:  read from stdin\n");
	fprintf(stderr,"  --atrack,           -t:  audio track selection\n");
	fprintf(stderr,"  --softaud,          -l:  decode audio by software\n");
	fprintf(stderr,"  --loop,             -r:  loop files\n");
	fprintf(stderr,"  --nopat,            -s:  try to find apid and vpid without PAT\n");
	exit(1);
}


void check_track(int ac3, av_settings *av)
{
	if (ac3) {
		if (av->a52_track < 0x80)
			av->a52_track += 0x80;
		if ((av->a52_track < 0x80) || (av->a52_track > 0x87)) {
			fprintf (stderr, "Invalid track number: %s\n", optarg);
			exit(1);
		}
	} else {
		if (av->mpx_track < AUDIO_STREAM_S)
			av->mpx_track += AUDIO_STREAM_S;
		if ((av->mpx_track < AUDIO_STREAM_S) || 
		    (av->mpx_track > AUDIO_STREAM_E) ) {
			fprintf (stderr, "Invalid track number: %s\n", optarg);
			exit(1);
		}
	}
}

void set_audiot(char *audiot, int fda, int soft, av_settings *av)
{ 
	if (soft || audioSetStreamtype(fda, audiot) < 0){
		fprintf(stderr,
			"Audio format %s not supported by hardware\n",
			audiot);
		if (!strcmp(audiot,"AC3") &&  
		    audioSetStreamtype(fda, "LPCM") >= 0){
#ifdef HAVE_LIBA52
			av->soft_ac3 = 1;
#else
			exit(1);
#endif
		} else if (!strcmp(audiot,"MP2") &&  
			   audioSetStreamtype(fda, "LPCM") >= 0){
#ifdef HAVE_LIBMAD
			av->soft_mpx = 1;
#else
			exit(1);
#endif
		}	
	} else if (!strcmp(audiot,"AC3") || !strcmp(audiot,"LPCM")){
		audioSetExtID(fda, av->a52_track);
		av->aud_ext = 1;
	}
	
#ifdef HAVE_LIBA52
	if (av->soft_ac3){
		fprintf(stderr,	"switching to software AC3 decoding\n");
		init_a52_buf();
		av->state = a52_init (0);
		
		av->aud_ext = 1;
		av->needs_demux = 1;
	}
#endif

#ifdef HAVE_LIBMAD
	if (av->soft_mpx){
		fprintf(stderr,	"switching to software MPEG audio decoding\n");
		init_mad();
		av->needs_demux = 1;
	}
#endif
	
	
}

int main(int argc, char **argv)
{
	int fda,fdv,fd_dvr;
	int filefd=0;
	int i, c;
	uint16_t vpid = 0;
	uint16_t apid = 0;
	char *inpt = "PS";
	char *audiot = "MP2";
	int dev =0;
       	int dvr = 1;
	int stin = 0;
	int soft = 0;
	char devnamev[80];
	char devnamea[80];
	char devnamedv[80];
	av_settings av;
	int ftype = C_NOPES;
	int fset = 0;

	init_av(&av);

	while (1) {
		int option_index = 0;
		static struct option long_options[] =
		{
		 {"input_stream", required_argument, NULL, 'i'},
		 {"audio_out", required_argument, NULL, 'e'},
		 {"video_pid", required_argument, NULL, 'v'},
		 {"audio_pid", required_argument, NULL, 'a'},
		 {"device", required_argument, NULL, 'd'},
		 {"no_dvr", no_argument, NULL, 'n'},
		 {"stdin", no_argument, NULL, 'o'},
		 {"nopat", no_argument, NULL, 's'},
		 {"softaud", no_argument, NULL, 'l'},
		 {"loop", no_argument, NULL, 'r'},
		 {"a52track", required_argument , NULL, 't'},
		 {"help", no_argument , NULL, 'h'},
		 {0, 0, 0, 0}
		};

		c = getopt_long (argc, argv, "li:hv:a:d:noe:st:r",
				 long_options, &option_index);
		if (c == -1)
			break;
		
		switch (c) {

		case 'i':
			inpt = optarg;
			fset = 1;
			break;

		case 'e':
			audiot = optarg;
			break;

		case 'v':
			vpid = atoi(optarg);
			inpt = "TS";
			fset = 1;
			break;

		case 'a':
			apid = atoi(optarg);
			inpt = "TS";
			fset = 1;
			break;

		case 'd':
			dev = atoi(optarg);
			break;

		case 'n':
			av.nopat = 1;
			dvr = 0;
			inpt = "TS";
			fset = 1;
			break;

		case 'o':
			printf("Reading from stdin\n");
			stin = 1;
			break;

		case 's':
			av.nopat = 1;
			inpt = "TS";
			fset = 1;
			break;

		case 'l':
			soft = 1;
			break;

		case 'r':
			loop = 1;
			break;

		case 't':
			av.aud_ext = 1;
			av.a52_track = atoi(optarg);
			av.mpx_track = atoi(optarg);
			break;

		case 'h':
		case '?':
		default:
			usage(argv[0]);
		}
	}

	if (optind){
		int ind=optind;
		while (ind < argc && ftype == C_NOPES){
			ftype = check_file_ts(argv[ind]);
			ind++;
		}
		if (ind > optind) optind = ind-1;
	} else ftype = check_fd_ts(STDIN_FILENO);

	switch(ftype){
	case C_NOPES:
		fprintf(stderr,"Can't determine filetype\n");
		fprintf(stderr,"Using %s\n", inpt);
		break;
		
	case C_MPEG:
		if (fset && strcmp(inpt,"PS")){
			fprintf(stderr,"Warning: filetype is not %s\n",
				inpt);
		} else {
			inpt = "PS";
		}
		break;
		
	case C_TS:
		if (fset && strcmp(inpt,"TS")){
			fprintf(stderr,"Warning: filetype is not %s\n",
				inpt);
		} else {
			if (!fset){
				av.nopat = 1;
				dvr = 0;
				inpt = "TS";
			}
		}
		break;
	}				

	if (!strcmp(audiot,"AC3")) check_track(1, &av);
	else check_track(0, &av);
	
	yskip = 0;
	if ( !strcmp(inpt, "PS")){
		if (open_av(&fdv, &fda, dev, 1) < 0){
			perror("open_av");
			exit(1);	
		}
		set_audiot(audiot, fda, soft, &av);

		if (videoSetStreamtype(fdv, "PROG") < 0){
			if (videoSetStreamtype(fda, "MPEG2") >= 0){
				av.needs_demux = 1;
			} else 	{
				fprintf(stderr,
					"Stream type not supported by hardware\n");
				exit(1);
			}	
		}

		if (stin) { 
			play_file_video_stdin(fdv, fda, &av);
		} else {
			curses_init();
			mvprintw(yskip++,0,"Welcome to NTuxplayer");
			yskip++;
			mvprintw(yskip,0,"p : playback");
			mvprintw(yskip++,30,"s : stop playback");
			mvprintw(yskip,0,"c : continue playback");
			mvprintw(yskip++,30,"z : freeze picture");
			mvprintw(yskip,0,"f : fast forward");
			mvprintw(yskip++,30,"m : slow motion");
			mvprintw(yskip,0,"+ : increase volume");
			mvprintw(yskip++,30,"- : decrease volume");
			mvprintw(yskip,0,"n : skip to next file or end");
			mvprintw(yskip++,30,"b : skip to previous file");
			mvprintw(yskip,0,"->: skip ahead");
			mvprintw(yskip++,30,"^ : big skip ahead");
			mvprintw(yskip,0,"<-: skip back");
			mvprintw(yskip++,30,"|v:  big skip back");
			mvprintw(yskip,0,"l : toggle loop");
			mvprintw(yskip++,30,"q : quit");
			if (av.aud_ext)
				mvprintw(yskip++,0,"1-8: audio track");

			
			yskip+=2;
			refresh();

			do{
				for ( i=optind; i< argc; i++){
					
					if ( i < optind ) i = optind;
					if (!fset && 
					    ftype != check_file_ts(argv[optind]))
						i++;
					if ( (filefd = open(argv[i],O_RDONLY)) < 0){
						perror("File open:");
						return -1;
					}
					mvprintw(yskip-1,0,"Playing %s      ",argv[i]);
					refresh();
					if (play_file_video(filefd, 
							    fdv, fda, i-optind, &av )
					    && i>=optind)
						i-=2;
					close(filefd);
				}
				videoClear(fdv);
				audioClear(fda);
				mute(fda, 0, &av);
			} while (loop);
		}
		if (fda) audioSetAVSync(fda, true);

		close(fdv);
		close(fda);
	} else if ( !strcmp(inpt, "TS") ){

		sprintf(devnamedv,DVR_DEV,dev);	
		sprintf(devnamev,VIDEO_DEV,dev);	
		sprintf(devnamea,AUDIO_DEV,dev);

		if((fd_dvr = open(devnamedv,O_WRONLY)) < 0){
			perror(devnamedv);
			return -1;
		}
	
		if((fdv = open(devnamev,O_RDWR|O_NONBLOCK)) < 0){
			perror(devnamev);
			return -1;
		}
	    
		if((fda = open(devnamea,O_RDWR|O_NONBLOCK)) < 0){
			perror(devnamea);
			return -1;
		}
		set_audiot(audiot, fda, soft, &av);

	
		switch (dvr){
		case 1:
			if (stin) { 
				play_file_dvr_stdin(fd_dvr, fda, fdv, apid, 
						    vpid, dev, &av);
			} else {
				curses_init();
				mvprintw(yskip++,0,"Welcome to NTuxplayer");
				yskip++;
				mvprintw(yskip,0,"c : continue playback");
				mvprintw(yskip++,30,"z : freeze picture");
				mvprintw(yskip,0,"f : fast forward");
				mvprintw(yskip++,30,"m : slow motion");
				mvprintw(yskip,0,"+ : increase volume");
				mvprintw(yskip++,30,"- : increase volume");
				mvprintw(yskip,0,"n : skip to next file or end");
				mvprintw(yskip++,30,"b : skip to previous file");
				mvprintw(yskip,0,"->: skip ahead");
				mvprintw(yskip++,30,"^ : big skip ahead");
				mvprintw(yskip,0,"<-: skip back");
				mvprintw(yskip++,30,"|v:  big skip back");
				mvprintw(yskip,0,"l : toggle loop");
				mvprintw(yskip++,30,"q: quit");

				yskip+=4;
				refresh();
				do{
					for ( i=optind; i< argc; i++){
						if ( i < optind ) i = optind;
						if (!fset && 
						    ftype != 
						    check_file_ts(argv[optind]))
							i++;
						if ( (filefd = 
						      open(argv[i],O_RDONLY)) < 0){
							perror(argv[i]);
							return -1;
						}
						mvprintw(yskip-3,0,"Playing %s        "
							 ,argv[i]);
						refresh();
						if (play_file_dvr(fd_dvr, fda, fdv, 
								  filefd, apid, 
								  vpid, dev, &av)
						    && i>=optind)
							i-=2;
						close(filefd);
					}
				} while (loop);
				videoClear(fdv);
				audioClear(fda);
				mute(fda, 0, &av);
			}
			break;
		case 0:
			if (videoSetStreamtype(fdv, "PROG") < 0){
				if (videoSetStreamtype(fda, "MPEG2") >= 0){
					av.needs_demux = 1;
				} else 	{
					fprintf(stderr,
						"Stream type not supported by hardware\n");
					exit(1);
				}
			}
	
			if (stin) { 
				play_ts_video_stdin(fdv, fda, apid, vpid,&av);
			} else {
				curses_init();
				mvprintw(yskip++,0,"Welcome to NTuxplayer");
				yskip++;
				mvprintw(yskip,0,"p : playback");
				mvprintw(yskip++,30,"s : stop playback");
				mvprintw(yskip,0,"c : continue playback");
				mvprintw(yskip++,30,"z : freeze picture");
				mvprintw(yskip,0,"f : fast forward");
				mvprintw(yskip++,30,"m : slow motion");
				mvprintw(yskip,0,"+ : increase volume");
				mvprintw(yskip++,30,"- : increase volume");
				mvprintw(yskip,0,"->: skip ahead");
				mvprintw(yskip++,30,"^ : big skip ahead");
				mvprintw(yskip,0,"<-: skip back");
				mvprintw(yskip++,30,"|v: big skip back");
				mvprintw(yskip,0,"n : skip to next file or end");
				mvprintw(yskip++,30,"b : skip to previous file");
				mvprintw(yskip,0,"l : toggle loop");
				mvprintw(yskip++,30,"q: quit");
				yskip+=3;
				refresh();

				do{
					for ( i=optind; i< argc; i++){
						if ( i < optind ) i = optind;
						if (!fset && 
						    ftype != 
						    check_file_ts(argv[optind]))
							i++;
						if ( (filefd = 
						      open(argv[i],O_RDONLY)) < 0){
							perror(argv[i]);
							return -1;
						}
						mvprintw(yskip-2,0,"Playing %s        "
						 ,argv[i]);
						refresh();
						if (play_ts_video(filefd, fdv, fda, 
								  apid, vpid, &av)
						    && i>=optind)
							i-=2;
						close(filefd);
					}
				} while (loop);
				
				videoClear(fdv);
				audioClear(fda);
				mute(fda, 0, &av);
			}
			if (fda) audioSetAVSync(fda, true);

			mute(fda, 0, &av);
			close(fdv);
			close(fda);
			break;
		}
	}

	return 0;


}

