#include "DVB.hh"

static void dvb2txt(char *out, char *in, int len)
{
        //char *chartable=0;
        unsigned char *txt=(unsigned char *) in;
        if (!len)
	        return;
	switch (*txt) {
	case 0x01 ... 0x0f:
	        txt++;
	        len--;
	        break;
	case 0x10:
	        txt+=3;
                len-=3;
	        break;
	}
	while (len) {
	        switch(*txt) {
		case 0x01 ... 0x1f:
		case 0x7f ... 0xa0:
		        len--;
			txt++;
			break;
		case 0x00:
		        len=1;
		default:
		        *(out++)=(char) *(txt++);
			len--;
			break;
		}
	}
}

DVB::~DVB() {
        delete [] lnbs;
	delete [] diseqcs;
	delete [] rotors;
	delete [] tps;
	delete [] chans;
	delete [] bouqs;
	delete [] sats;
	delete [] swis;
	delete [] ntws;
	close(fd_frontend);
	close(fd_demuxa);
	close(fd_demuxv);
	close(fd_demuxpcr);
	close(fd_demuxtt);
	close(fdvb);
}

void DVB::init(char *dvbname, char *siname, int min) {
	struct dvb_frontend_info feinfo;

	int failed =0;
	 
	minor = min;

	for (int i=0; i<NK; i++)
	        num[i]=0;

	lnbs   =new Lnb[maxs[LNB]];
	diseqcs=new DiSEqC[maxs[DIS]];
	rotors =new Rotor[maxs[ROTOR]];
	tps    =new Transponder[maxs[TRANS]];
	chans  =new Channel[maxs[CHAN]];
	bouqs  =new Bouquet[maxs[BOU]];
	sats   =new Sat[maxs[SAT]];
	swis   =new Switch[maxs[SWI]];
	ntws   =new Network[maxs[NTW]];
	if (minor < 0) return;

        if (fd_frontend > 0) close(fd_frontend);
        if (fd_demuxa > 0) close(fd_demuxa);
        if (fd_demuxv > 0) close(fd_demuxv);
        if (fd_demuxpcr > 0) close(fd_demuxpcr);
        if (fd_demuxtt > 0) close(fd_demuxtt);
	char devname[80];


	dvr_enabled = 0;

	sprintf(devname,"%s%d%s",ADAPTER,minor,FRONT_DEV);
	fd_frontend=open(devname, O_RDWR);
	if (fd_frontend < 0) {
		cerr << "Could not open " << devname << endl;
		front_type=-1;
		perror(devname);
		fd_frontend = -1;
		failed = 1;
	}
	ioctl(fd_frontend, FE_GET_INFO, &feinfo);
	front_type=feinfo.type;

	sprintf(devname,"%s%d%s",ADAPTER,minor,DEMUX_DEV);
	fd_demuxtt=open(devname, O_RDWR);
	if (fd_demuxtt < 0) {
		cerr << "Could not open " << devname << endl;
		perror(devname);
		fd_demuxtt = -1;
		failed = 1;
	}

	fd_demuxa=open(devname, O_RDWR);
	if (fd_demuxa < 0) {
		cerr << "Could not open " << devname << endl;
		perror(devname);
		fd_demuxa = -1;
		failed = 1;
	}

	fd_demuxpcr=open(devname, O_RDWR);
	if (fd_demuxpcr < 0) {
		cerr << "Could not open " << devname << endl;
		perror(devname);
		fd_demuxpcr = -1;
		failed = 1;
	}

	fd_demuxv=open(devname, O_RDWR);
	if (fd_demuxv < 0) {
		cerr << "Could not open " << devname << endl;
		perror(devname);
		fd_demuxv = -1;
		failed = 1;
	}
}

int DVB::SetTP(unsigned int tpid, unsigned int satid)
{
        int i;
	Transponder *tp=0;
	Sat *sat=0;
	Lnb *lnb=0;

	for (i=0; i<num[TRANS]; i++) {
	        if (tps[i].id==tpid && tps[i].satid==satid) {
		        tp=&tps[i];
			break;
		}
	}

	if (!tp) {
		fprintf(stderr,"Transponder not found!\n");
	        return -1;
	}

	for (i=0; i<num[SAT]; i++) {
		if (sats[i].id==tp->satid) {
		        sat=&sats[i];
			break;
		}
	}

	if (!sat){
		fprintf(stderr,"Satellite not found!\n");
	        return -1;
	}

	for (i=0; i<num[LNB]; i++)  
               if (lnbs[i].id==sat->lnbid) {
		        lnb=&lnbs[i];
			break;
	       }

	if (!lnb){
		fprintf(stderr,"LNB not found!\n");
	        return -1;
	}



	switch (front_type) {
	case FE_QPSK:
	        if (tp->freq < lnb->slof) {
		        front_param.frequency = (tp->freq - lnb->lof1);
			tone = SEC_TONE_OFF;
		} else {
		        front_param.frequency = (tp->freq - lnb->lof2);
			tone = SEC_TONE_ON;
		}
		if (tp->pol) voltage = SEC_VOLTAGE_18;
		else voltage = SEC_VOLTAGE_13;
		set_diseqc_nb(lnb->diseqcnr);
		front_param.u.qpsk.symbol_rate = tp->srate;
		front_param.u.qpsk.fec_inner = (fe_code_rate_t)tp->fec;
		front_param.inversion = INVERSION_OFF;
		break;
	case FE_QAM:
	        front_param.frequency = tp->freq;
		front_param.inversion = INVERSION_AUTO;
		front_param.u.qam.symbol_rate = tp->srate;
		front_param.u.qam.fec_inner = (fe_code_rate_t)tp->fec;
		front_param.u.qam.modulation=(fe_modulation_t) (tp->qam+1);
	        break;

	case FE_OFDM:
	        front_param.frequency = tp->freq;
		front_param.inversion = INVERSION_AUTO;
		front_param.u.ofdm.bandwidth = (fe_bandwidth_t)tp->band;
		front_param.u.ofdm.code_rate_HP = (fe_code_rate_t)tp->hp_rate;
		front_param.u.ofdm.code_rate_LP = (fe_code_rate_t)tp->lp_rate;
		front_param.u.ofdm.constellation = (fe_modulation_t)tp->mod;
		front_param.u.ofdm.transmission_mode = 
			(fe_transmit_mode_t)tp->transmode;
		front_param.u.ofdm.guard_interval = (fe_guard_interval_t)tp->guard;
		front_param.u.ofdm.hierarchy_information = 
			(fe_hierarchy_t)tp->hierarchy;
		break;
	}
        return 0;
}

static uint16_t get_pid(uint8_t *pid)
{
        uint16_t pp;

        pp = (pid[0] & 0x1f) << 8;
        pp |= pid[1] &0xff;
        return pp;
}



void DVB::check_all_pids()
{
	for (int i = 0; i < num[CHAN]; i++){
		cerr << "checking " << chans[i].name << endl;
		SetChannel(i); 
	}
}

int DVB::check_pids(Channel *chan)
{
	uint8_t *prog;
	int slen, i, nprog, ilen, c;
	uint16_t prog_pid = 0, pnr = 0;
	int found = 0;
	int es_length;
	int e;
	int oldnum;

	unsigned char buf[MAXSECSIZE], msec=0, sec=0;
	
	oldnum=chan->apidnum;
	time_t count = time(0)+2;
	while (sec<=msec && !found && count > time(0)) {
		if (GetSection(buf, 0, 0, sec, msec)>0 && !buf[0]){
			sec++;
			slen = (((buf[1]&0x0f) << 8) | buf[2]) +3;
			nprog = (slen - 12) / 4;
			prog = buf+ 8;
			for (i = 0; i < nprog; i++){
				pnr = (prog[0] << 8)|prog[1];
			  if ( chan->pnr == pnr ){
					prog_pid = get_pid(prog+2);
					found = 1;
					break;
				} 
				prog += 4;
			} 
		}
	}
	if (!found) return -1;
	chan->apidnum = 0;
	sec = 0;
	msec = 0;
	
	while (sec<=msec && count > time(0)) {
		if (GetSection(buf,prog_pid, 2, sec, msec)>0){
			sec++;
			slen = (((buf[1]&0x0f) << 8) | buf[2]) +3;
			ilen = ((buf[10]&0x03)  <<8) | buf[11];
			chan->pcrpid =  get_pid(buf+8);
			c = 12+ilen;
			while (c < slen){
				if(count < time(0)) break;
				switch (buf[c]) {
				case 1: 
				case 2: 
					if (chan->vpid == NOPID)
						chan->vpid = get_pid(buf+c+1);
					break;
				case 3: 
				case 4: 
				{
					int afound = 0;
					uint16_t apid = get_pid(buf+c+1); 
					if (chan->apidnum>=MAXAPIDS) {
						cerr << "Need more apids\n";
						break;
					}
					
					for (int i = 0; 
					     i < chan->apidnum; i++){
						if ( apid ==
						     chan->apids[i]){
							afound = 1;
							break;
						}
					}
					if (! afound){
						chan->apids[chan->apidnum++]=
							apid;
					
						es_length = (buf[c+3]&0x0F) 
							<< 8;
						es_length |= buf[c+4];

						e = 0;
						while (buf[c+5+e] != 0x0a && 
							c+5+e < es_length) e++;
						if (buf[c+6+e] == 0x04){
							memcpy(chan->apids_name
							       +(chan->apidnum
								 -1)*4,
							       buf+c+7+e, 3);
							chan->apids_name[
								(chan->apidnum-1)*4+3] =
								'\0';
						}
					}
					break; 
				}
				case 6: 
					chan->ttpid = get_pid(buf+c+1);
					break;
				default:
					break;
				}
				uint16_t silen = (((buf[c+3] & 0x0f) << 8) | 
						  buf[c+4]);
				c += 5 + silen;
			}
		}
	}
	if (!chan->apidnum)
		chan->apidnum=oldnum;
	chan->checked = 1;
	return 0;
}

int DVB::get_all_progs(uint16_t *progbuf, uint16_t *pnrbuf, int length)
{
	int nfound = 0, oldone = 0, i, j;
	unsigned char buf[MAXSECSIZE], msec=0, sec=0;
	uint16_t prog_pid = 0, pnr = 0;
	int slen, nprog;
	uint8_t *prog;

	time_t count = time(0)+3;
	while (sec<=msec && count > time(0)) {
		if (GetSection(buf, 0, 0, sec, msec)>0 && !buf[0]){
			sec++;
			slen = (((buf[1]&0x0f) << 8) | buf[2]) +3;
			nprog = (slen - 12) / 4;
			prog = buf+ 8;
			for (i = 0; i < nprog; i++){
				pnr = (prog[0] << 8)|prog[1];
				prog_pid = get_pid(prog+2);
				
				oldone = 0;
				for(j=0; j<nfound; j++)
					if (pnr == pnrbuf[j]) oldone = 1;
					
				if ( !oldone && nfound < length ){
					pnrbuf[nfound]=pnr;
					progbuf[nfound]=prog_pid;
					nfound++;
				} else return -1;
				prog += 4;
			} 
		}
	}
	return nfound;
}

int DVB::get_pids(uint16_t prog_pid, uint16_t *vpid, uint16_t *apids)
{
	int slen, ilen, c;
	int found = 0;
	unsigned char buf[MAXSECSIZE], msec=0, sec=0;
	int apidnum = 0;
	
	time_t count = time(0)+2;

	sec = 0;
	msec = 0;
	*vpid = NOPID;
	while (sec<=msec && count > time(0)) {
		if (GetSection(buf,prog_pid, 2, sec, msec)>0){
			sec++;
			slen = (((buf[1]&0x0f) << 8) | buf[2]) +3;
			ilen = ((buf[10]&0x03)  <<8) | buf[11];
			c = 12+ilen;
			while (c < slen){
				if(count < time(0)) break;
				switch (buf[c]) {
				case 1: 
				case 2: 
					*vpid = get_pid(buf+c+1);
					found = 1;
					return found;
					break;
				case 3: 
				case 4: 
				{
					int afound = 0;
					uint16_t apid = get_pid(buf+c+1); 
					if (apidnum >= MAXAPIDS) {
						cerr << "Need more apids\n";
						break;
					}
					
					for (int i = 0; i < apidnum; i++){
						if ( apid == apids[i]){
							afound = 1;
							break;
						}
					}
					if (! afound){
						apids[apidnum++] = apid;
					}
					break; 
				}
				case 6: 
					break;
				default:
					break;
				}
				uint16_t silen = (((buf[c+3] & 0x0f) << 8) | 
						  buf[c+4]);
				c += 5 + silen;
			}
		}
	}

	return found+apidnum;
}

uint16_t DVB::find_pnr(uint16_t svpid, uint16_t sapid)
{
	int nfound = 0;
	int pfound = 0;
	uint16_t progs[100];
	uint16_t pnrs[100];
	uint16_t vpid;
	uint16_t apids[MAXAPIDS];

	nfound = get_all_progs(progs, pnrs, 100);
	for(int i=0; i < nfound; i++){
		if ( (pfound = get_pids(progs[i], &vpid, apids)) ){
			if(svpid != NOPID && vpid == svpid ) return pnrs[i];
			else if ( svpid == NOPID ){
				for (int j=0; j<pfound; j++){
					if (apids[j] == sapid) return pnrs[i];
				}
			}
		}
	}
	return 0;
}



void DVB::parse_sdt(uint8_t *buf, int count, Channel *chan)
{
	int c = 0;
	uint8_t dtag;
	uint8_t dlen;
	uint8_t slen=0;
	int i;
	char name[MAXNAM];

	while ( c < count ){
		dtag = buf[c++];
		dlen = buf[c++];
		while ( dlen ){
			switch ( dtag ){
			case 0x09: /* ca desc */
				chan->type = 1;
				dlen = 0;
				c+= dlen;
				break;

			case 0x48: /* service desc */
				i = c;
				i++; /* service type */
				i += buf[i]+1;  /* provider name */
				slen = buf[i++];
				memcpy(name,(char*)(&buf[i]), int(slen));
				name[slen]=0;
				memset(chan->name, 0, MAXNAM);
				dvb2txt(chan->name,name, int(slen));
				i+= slen;
				c+= dlen;
				dlen = 0;
				break;
			default:
				c+= dlen;
				dlen = 0;
			}
		}

	}

}

void DVB::scan_sdt(Channel *chan)
{
	int slen, ilen, c;
	unsigned char buf[MAXSECSIZE], msec=0, sec=0;
	uint16_t pnr;
	uint8_t ca;

	time_t count = time(0)+5;
	while (sec<=msec && count > time(0)) {
		if (GetSection(buf, 0x11, 0x42, sec, msec)>0){
			sec++;
			slen = (((buf[1]&0x0f) << 8) | buf[2]) +3;
			c = 11;
			while (c < slen-4){
				pnr = (buf[c]<<8)|buf[c+1];
				chan->has_eit = -1;
				chan->pres_follow = -1;
				if (buf[c+2] & 0x02) chan->has_eit = 0;
				if (buf[c+2] & 0x01) chan->pres_follow = 0;
				c+=3;
				ca=(buf[c]&0x10)>>4;
				ilen=((buf[c]&0x0f)<<8)|buf[c+1];
				c+=2;
				if ( chan->pnr == pnr ){
					parse_sdt(&buf[c], ilen, chan);
				}
				c+=ilen;
			}
		}
	}
}

void eit_cb(uint8_t *buf, int l, int pnr, int c_n)
{
	cout << "Type: " << c_n << "  PNR:" << pnr << endl;
	for (int i=0; i < l/16+1; i++){
		cout << "0x" << HEX(4) << i << dec << "  ";
		for (int j=0; j < 16; j++){
			if (j+i*16 < l)
				cout << HEX(2) << int(buf[j+i*16]) << dec <<
					" ";
			else cout << "   ";
		}
		for (int j=0; j < 16 && j+i*16 < l; j++){
			uint8_t c = (char)buf[j+i*16];
			switch ((int)c) {
			case 0x01 ... 0x1f:
			case 0x7f ... 0xa0:
			case 0x00:
				cout << "." ;
				break;
			default:
				cout << char(c) ;
				break;
			}
		}
		cout << endl;
	}
	cout << endl;
}

void DVB::scan_pf_eit(Channel *chan, 
		      void (*callback)(uint8_t *data, int l, int pnr, int c_n) 
		      = eit_cb)
{
	int slen, ilen, c;
	unsigned char buf[MAXSECSIZE], msec=0, sec=0;
	uint16_t pnr = NOID;
	int8_t ver;
	int c_n;

	time_t count = time(0)+20;
	while ( chan->pnr != pnr && count > time(0)) {
		if (GetSection(buf, 0x12, 0x4E, sec, msec)>0){
			sec++;
			c = 1;
			slen = (((buf[c]&0x0f) << 8) | buf[c+1]) +3;
			c += 2;
			pnr = (buf[c]<<8)|buf[c+1];
			c += 2;
			ver = buf[c] & 0x3E;
			c_n = buf[c] & 0x01;
			c += 8;
			while (c < slen-4){
				c += 10;
				ilen=((buf[c]&0x0f)<<8)|buf[c+1];
				c+=2;
				callback(buf+c, ilen, pnr, c_n);
				c+=ilen;
			}
		}				
	}
}

void DVB::scan_pf_eit(int chnr,
		      void (*callback)(uint8_t *data, int l, int pnr, int c_n) 
		      = eit_cb) 
{
	if (chnr>=num[CHAN] || chnr < 0) return;
	scan_pf_eit(&chans[chnr], callback);
}

void DVB::scan_pf_eit(int chnr) 
{
	if (chnr>=num[CHAN] || chnr < 0) return;
	scan_pf_eit(&chans[chnr]);
}


int DVB::SetChannel(Channel *chan,  char* apref, uint16_t *apidp, 
		    uint16_t *vpidp) 
{
	int i;
	int scan = 0;
	uint16_t apid=0, vpid=0;

	if (chan->pnr == NOPID && (chan->vpid != NOPID || 
				   chan->apids[0] != NOPID))
		chan->pnr = find_pnr(chan->vpid, chan->apids[0]);
	if (chan->vpid == NOPID && chan->apidnum == 0 ) 
		if (!chan->checked){ 
			check_pids(chan);
			scan = 1;
		}
	
	vpid = chan->vpid;
	if (chan->apidnum){
		if (apref){
			int found = 0;
			for (i=0; i < chan->apidnum; i++){
				if (!strncasecmp(chan->apids_name+4*i,
						 apref,3)){
					found=1;
					break;
				}
			}
			if (found) apid = chan->apids[i];
			else apid = chan->apids[0];
		} else apid = chan->apids[0];
	}

	if (vpid != NOPID) set_vpid(vpid);	
	set_apid(apid);
	set_pcrpid(chan->pcrpid);
	if (chan->ttpid != NOPID) set_ttpid(chan->ttpid);
	
	if (scan)
		scan_sdt(chan);

	if (fdvb >= 0){
		struct tm *lt;
		time_t t;
		
		
		t = time(0);
		lt = localtime(&t);
		
		ostrstream fstr;
		osd.Clear();
		
		fstr << setw(2) << setfill('0') << lt->tm_hour 
		     << ":" << setw(2) << setfill('0') 
		     << lt->tm_min << "  "
		     << chan->name <<  ends;
		
		osd.Text(0, 0, 0, 1, fstr.str());
		osd.Show();
	}
	if (vpidp) *vpidp = vpid;
	if (apidp) *apidp = apid;
//	scan_pf_eit(chan);

	if (vpid==NOPID && apid == NOPID) return -1;
	check_frontend();
        return 0;

}

int DVB::SetChannel(int chnr, char* apref, uint16_t *apidp, 
		    uint16_t *vpidp) 
{
	if (chnr>=num[CHAN] || chnr < 0)
	        return -1;
	get_front();
	if (SetTP(chans[chnr].tpid, chans[chnr].satid) < 0) return -1;
	set_front();
       

	return SetChannel(&chans[chnr], apref, apidp, vpidp);
}



int DVB::SetChannel(uint16_t sid, uint16_t onid, 
		    uint16_t tpid, uint16_t satid)
{
        int chnr;
	Channel *chan=0;

	for (chnr=0; chnr<num[CHAN]; chnr++)  
	        if (chans[chnr].pnr==sid   && 
		    (onid==NOID  || chans[chnr].onid==onid) && 
		    (satid==NOID || chans[chnr].satid==satid)  && 
		    (tpid==NOPID  || chans[chnr].tpid==tpid)
		    ) {
		        chan=&chans[chnr];
			break;
		}
	if (!chan)
	        return -1;

	if (tpid==NOID)
 	        tpid=chan->tpid;
	if (onid==NOID)
 	        onid=chan->onid;
	if (satid==NOID)
 	        satid=chan->satid;

	get_front();
	if (SetTP(tpid, satid) < 0) return -1;
	set_front();

        set_vpid(chan->vpid);
        set_apid(chan->apids[0]);
	set_ttpid(chan->ttpid);
	set_pcrpid(chan->pcrpid);

	return chnr;
}

int DVB::GetChannel(int chnr, struct channel *) 
{
        int i;
	Channel *chan=0;
	Transponder *tp=0;
	Sat *sat=0;
	Lnb *lnb=0;

	if (chnr>=num[CHAN])
	        return -1;
	chan=&chans[chnr];
	
	for (i=0; i<num[TRANS]; i++)  
               if (tps[i].id==chan->tpid) {
		        tp=&tps[i];
			break;
	       }
	if (!tp)
	        return -1;

	for (i=0; i<num[SAT]; i++)  
               if (sats[i].id==tp->satid) {
		        sat=&sats[i];
			break;
	       }
	if (!sat)
	        return -1;

	for (i=0; i<num[LNB]; i++)  
               if (lnbs[i].id==sat->lnbid) {
		        lnb=&lnbs[i];
			break;
	       }
	if (!lnb)
	        return -1;
	set_front();

        return 0;

}

int DVB::AddLNB(int id, int t, uint l1, uint l2, uint sl,
		int dnr, int dis, int sw)
{
        if (num[LNB]>=maxs[LNB])
	        return -1;
	for (int i=0; i< num[LNB]; i++){
		if (lnbs[i].id == id && lnbs[i].diseqcnr == dnr){
			cerr << "Warning: LNB already defined:" << endl;
			cerr << "ID: " << id << "  DISEQCNR: " << dnr << endl;
			return -1;
		}
	}

	lnbs[num[LNB]].init(t, l1, l2, sl, dnr, dis, sw);
	lnbs[num[LNB]].id=id;
	num[LNB]++;
	return 0;
}



int DVB::AddTP(Transponder &tp)
{
        if (num[TRANS]>=maxs[TRANS])
	        return -1;

	if (tp.id == NOID){
		max_tpid++;
	        tp.id = max_tpid;
	} else if (tp.id > max_tpid) max_tpid = tp.id;

	for (int i=0; i<num[TRANS]; i++)
		if (tps[i].id==tp.id && tps[i].satid == tp.satid){
			cerr << "Warning: TP already defined:" 
			     << endl;
			cerr << "ID: " << hex << tp.id ;
			cerr << "  SATID: "<< hex  << tp.satid;
			cerr << endl;
			return i;
		} 
			
	tps[num[TRANS]]=tp;
	num[TRANS]++;
	return num[TRANS]-1;
}

void DVB::find_satid(Channel &chan)
{
	for (int i=num[TRANS]; i >= 0; i--){
		if (chan.tpid == tps[i].id){
			chan.satid = tps[i].satid;
			return;
		}
	}
}

int DVB::AddChannel(Channel &chan)
{
        int i;

        if (num[CHAN]>=maxs[CHAN])
	        return -1;
	
	if ( chan.satid == NOID)
		find_satid(chan);

        for (i=0; i<num[CHAN]; i++) {
		if (chan.pnr != NOPID && chan.pnr == chans[i].pnr && 
		    chan.satid == chans[i].satid
			&& chan.tpid == chans[i].tpid) {
		        cerr << "Channel " << chan.name << " ("
			     << hex << chan.pnr << ") exists" << endl;
			return i;
		}
		if (chan.pnr == NOPID && chan.vpid == chans[i].vpid &&
		    chan.apids[0] == chans[i].apids[0] &&
		    chan.satid == chans[i].satid
		    && chan.tpid == chans[i].tpid) {
		        cerr << "Channel " << chan.name << " ("
			     << hex << chan.pnr << ") exists" << endl;
			return i;
		}
	}

	chan.id = num[CHAN];
	chans[num[CHAN]] = chan;
	num[CHAN]++;
	return chan.id;
}

int DVB::AddSat(Sat &sat)
{
        int i;

        if (num[SAT]>=maxs[SAT])
	        return -1;
	if (!sat.id)
	        sat.id=num[SAT];

        for (i=0; i<num[SAT]; i++) {
 	        if (sat.lnbid == sats[i].lnbid) {
		        cerr << "Sat exists\n";
			return i;
		}
	}
	sats[num[SAT]]=sat;
	num[SAT]++;
	return sat.id;
}

int DVB::AddSat(int id, unsigned int lnbid, char *name, uint fmin, uint fmax)
{
        int i,j;

	if (num[SAT]==maxs[SAT]) 
	        return -1;
	for (i=0; i<num[LNB]; i++) {
	  if (lnbs[i].id==lnbid) {
		  for (j=0; j<num[SAT]; j++) {
			  if (lnbid == sats[j].lnbid) {
				  cerr << "Sat exists\n";
				  return j;
			  }
		  }
		  sats[num[SAT]].id=id;
		  sats[num[SAT]].lnb=&lnbs[i];
		  sats[num[SAT]].lnbid=lnbs[i].id;
		  strncpy(sats[num[SAT]].name, name, maxname);
		  sats[num[SAT]].name[maxname]=0;
		  sats[num[SAT]].fmin=fmin;
		  sats[num[SAT]].fmax=fmax;
		  num[SAT]++;
		  return num[SAT]-1;
		}
	}
	return -1;

}


int DVB::search_in_TP(uint16_t tpid, uint16_t satid)
{
	int nfound = 0;
	int pfound = 0;
	uint16_t progs[100];
	uint16_t pnrs[100];
	uint16_t vpid;
	uint16_t apids[MAXAPIDS];
	int n=0;
	
	get_front();
	if (SetTP(tpid, satid) < 0) return -1;
	set_front();

	nfound = get_all_progs(progs, pnrs, 100);
	for(int i=0; i < nfound; i++){
		vpid = NOPID;
		if ( (pfound = get_pids(progs[i], &vpid, apids)) &&
			vpid != NOPID){
			Channel chan;
		       
			chan.pnr = pnrs[i];
			chan.satid = satid;
			chan.tpid = tpid;
			chan.vpid = NOPID;
			chan.apidnum = 0;
			chan.checked = 0; 
			chan.ttpid = NOPID;

			SetChannel(&chan);
			sleep(2);
			if (AddChannel(chan)==num[CHAN]-1){
				n++;
			}		
		}
	}
	return n;
}

int DVB::search_in_TP(Transponder &tp)
{
	return search_in_TP(tp.id, tp.satid);
}

ostream &operator<<(ostream &stream, DVB &x) {
	int i,j,k,l;
	
	for (i=0; i<x.num[LNB]; i++) {
		stream << x.lnbs[i];
		
		for (j=0; j<x.num[SAT]; j++)
			if (x.sats[j].lnbid==x.lnbs[i].id) {
				stream << x.sats[j];
				
				for (k=0; k<x.num[TRANS]; k++) 
					if (x.tps[k].satid==x.sats[j].id) {
						x.tps[k].type = x.front_type;
						stream << x.tps[k];
						
						for (l=0; l<x.num[CHAN]; l++) 
							if (x.chans[l].tpid==
							    x.tps[k].id && 
							    x.chans[l].satid == x.tps[k].satid )
								stream << x.chans[l];
					}
			}
	}
	return stream;
}



int DVB::check_input_format(istream &ins)
{
	streampos pos = ins.tellg();
	int found = 0;
	char *test_keys[]={
		"LNB","DISEQC","ROTOR","TRANSPONDER","CHANNEL",
		"BOUQUET","SAT","SWITCH","NETWORK","<?xml", ":SAT", NULL
	};
	enum { XML_START=9, NOKIA_START};

	int f = -1;
	while(!found && !ins.eof()){
		char keybuf[MAXNAM];
		ins >> keybuf;
		int n=findkey(keybuf, test_keys);
		
		switch (n){
		case LNB:
		case DIS:
		case ROTOR:
		case TRANS:
		case CHAN:
		case BOU:
		case SAT:
			found = 1;
			f = DVB_ORIG;
			break;
			
		case NOKIA_START:
			found = 1;
			f = DVB_NOKIA;
			break;
			
		case XML_START:
			found = 1;
			f = DVB_XML;
			break;
		default:
			cerr << "Error: " << keybuf 
			     << " is not a valid keyword at " 
			     << endl;
			exit(0);
			
		}
	} 
	ins.seekg(pos);
	return f;
}

void DVB::read_original(istream &ins)
{
	char *names[] = {
		"LNB","DISEQC","ROTOR","TRANSPONDER","CHANNEL",
		"BOUQUET","SAT","SWITCH","NETWORK", NULL
	};

	cerr << "Reading original format" << endl;
	while(!ins.eof()){
		char keybuf[MAXNAM];
		ins >> keybuf;
		int n=findkey(keybuf, names);
		if (n<0) {
			cerr << "Error: " << keybuf 
			     << " is not a valid keyword at " 
			     << endl;
			exit(0);
		} else {
			if (num[n]< maxs[n]){
				switch (n){
				case LNB:
				{
					Lnb lnb;
					lnb.name[0]='\0';
					ins >> lnb;
					AddLNB(lnb.id, lnb.type, lnb.lof1, 
						 lnb.lof2, lnb.slof, 
						 lnb.diseqcnr, lnb.diseqcid, 
						 lnb.swiid);
					break;
				}
				case DIS:
					num[n]++;
					ins >> diseqcs[num[n]-1];
					break;
				case ROTOR:
					num[n]++;
					ins >> rotors[num[n]-1];
					break;
				case TRANS:
				{
					Transponder tp;
					ins >> tp;
					AddTP(tp);
					break;
				}
				case CHAN:
				{
					Channel chan;
					ins >> chan;
					AddChannel(chan);
					break;
				}
				case BOU:
					num[n]++;
					ins >> bouqs[num[n]-1];
					break;
				case SAT:
				{
					Sat sat;
					ins >> sat;
					AddSat(sat);
					break;
				}
				}
			} else {
				cerr << "not enough channels" << endl;
				break;
			}
		}
	} 
	
}

istream &operator>>(istream &ins, DVB &x)
{
	int format = x.check_input_format(ins);

	switch(format){
	case DVB_ORIG:
		x.read_original(ins);
		break;

	case DVB_NOKIA:
	{
		nokiaconv cc(&x);

		cc.lnb_sat.n = 4;
		cc.lnb_sat.diseqc[0] = 0;
		cc.lnb_sat.diseqc[1] = 1;
		cc.lnb_sat.diseqc[2] = 2;
		cc.lnb_sat.diseqc[3] = 3;
		strcpy(cc.lnb_sat.sat_names[0],"Astra");
		cc.lnb_sat.satid[0]=0x0192;
		strcpy(cc.lnb_sat.sat_names[1],"HotBird");
		cc.lnb_sat.satid[1]=0x0130;
		strcpy(cc.lnb_sat.sat_names[2],"Sirius");
		cc.lnb_sat.satid[2]=0x0050;
		cerr << "Reading NOKIA format" << endl;
		
		ins >> cc;
		break;
	}

	case DVB_XML:
	{
		xmlconv cc(&x);

		cc.lnb_sat.n = 4;
		cc.lnb_sat.diseqc[0] = 0;
		cc.lnb_sat.diseqc[1] = 1;
		cc.lnb_sat.diseqc[2] = 2;
		cc.lnb_sat.diseqc[3] = 3;
		strcpy(cc.lnb_sat.sat_names[0],"Astra");
		cc.lnb_sat.satid[0]=0x0192;
		strcpy(cc.lnb_sat.sat_names[1],"HotBird");
		cc.lnb_sat.satid[1]=0x0130;
		strcpy(cc.lnb_sat.sat_names[2],"Sirius");
		cc.lnb_sat.satid[2]=0x0050;
		cerr << "Reading XML format" << endl;
		
		ins >> cc;
		break;
	}
	default:
		cerr << "Unknown format. Can't open dvbrc. Exiting" << endl;
		exit(1);

	}
	return ins;
}




void hdump(unsigned char *buf, int n) 
{
	int i;
	
	for (i=0; i<n; i++)
		cerr << HEX(2) << (int) buf[i] << " ";
	cerr << endl;
}


void DVB::bar2(int x, int y, int w, int h, int val, int col1, int col2) 
{
	int sep=(w*val)>>16;
	
	if (fdvb >= 0) {
		osd.FillBlock(x, y, x+w-1-sep, y+h-1, col1);
		osd.FillBlock(x+w-1-sep, y, 515, y+h-1, col2);
	}
}

uint16_t DVB::SetFilter(uint16_t pid, uint16_t section, uint16_t mode) 
{ 
	struct dmx_sct_filter_params secFilterParams;
	char devname[80];
	sprintf(devname,"%s%d%s",ADAPTER,minor,DEMUX_DEV);

	int fd_section=open(devname, O_RDWR);
	
	secFilterParams.pid=pid;
	memset(&secFilterParams.filter.filter, 0, DMX_FILTER_SIZE);
	memset(&secFilterParams.filter.mask, 0, DMX_FILTER_SIZE);
	memset(&secFilterParams.filter.mode, 0, DMX_FILTER_SIZE);
	secFilterParams.timeout = 0;
	secFilterParams.flags=DMX_IMMEDIATE_START;
	
	secFilterParams.filter.filter[0]=(section>>8)&0xff;
	secFilterParams.filter.mask[0]=section&0xff;
	
	if (ioctl(fd_section, DMX_SET_FILTER, &secFilterParams) < 0)  
		return 0xffff;
	return fd_section;
} 

int DVB::SetFilter(uint16_t pid, unsigned char *filter, 
		   unsigned char *mask,
		   uint32_t timeout, uint32_t flags) 
{
	
	struct dmx_sct_filter_params secFilterParams;
	char devname[80];
	sprintf(devname,"%s%d%s",ADAPTER,minor,DEMUX_DEV);

	int fd_section=open(devname, O_RDWR);
	
	secFilterParams.pid=pid;
	memset(&secFilterParams.filter.filter, 0, DMX_FILTER_SIZE);
	memset(&secFilterParams.filter.mask, 0, DMX_FILTER_SIZE);
	memset(&secFilterParams.filter.mode, 0, DMX_FILTER_SIZE);
	secFilterParams.timeout = timeout;
	secFilterParams.flags=DMX_IMMEDIATE_START;
	
	for (int i = 0; i < DMX_FILTER_SIZE; i++){
		secFilterParams.filter.filter[i]=filter[i];
		secFilterParams.filter.mask[i]=mask[i];
	}
	if (ioctl(fd_section, DMX_SET_FILTER, &secFilterParams) < 0)  
		return 0xffff;
	return fd_section;
}


int DVB::CloseFilter(int h) 
{
	close(h);
	return 0;
}


int DVB::GetSection(unsigned char *buf, ushort PID, unsigned char sec,
		    unsigned char secnum, unsigned char &msecnum) 
{
	int seclen=0;
	uint16_t handle, pid;
	unsigned char section, sectionnum=0xff, maxsec=0;
	struct pollfd pfd;
	int loopc = 0;

	if ((handle=SetFilter(PID, (sec<<8)|0x00ff, 0))==0xffff)
		return -1;
	do {
		seclen=0;
		pfd.fd = handle;
		pfd.events=POLLIN;
		if (poll(&pfd, 1, 2000)==0) {
//			cerr << "Timeout\n";
			break;
		}
		loopc++;
		
		pid = PID;
		read(handle, buf, 3);
		seclen = 0;
		seclen |= ((buf[1] & 0x0F) << 8); 
		seclen |= (buf[2] & 0xFF);
		
		read(handle, buf+3, seclen);
		seclen+=3;
		section=buf[0];
		sectionnum=buf[6];
		maxsec=buf[7];
	} while ( loopc < maxsec*2 && 
		 (section != sec || pid != PID || sectionnum != secnum));
	msecnum=maxsec;
	CloseFilter(handle);
	return seclen;
}

int DVB::GetSection(uint8_t *buf, 
		    uint16_t PID, uint8_t *filter, uint8_t *mask,
		    uint8_t secnum, uint8_t &msecnum) 
{
	int seclen=0;
	uint16_t handle, pid;
	uint8_t section, sectionnum=0xff, maxsec=0;
	struct pollfd pfd;
	int loopc = 0;

	if ((handle=SetFilter(PID, filter, mask, 0, 0))==0xffff)
		return -1;
	do {
		seclen=0;
		pfd.fd=handle;
		pfd.events=POLLIN;
		if (poll(&pfd, 1, 20000)==0)
			break;
		loopc++;
		// read header
		pid = PID;
		read(handle, buf, 3);
		seclen = 0;
		seclen |= ((buf[1] & 0x0F) << 8); 
		seclen |= (buf[2] & 0xFF);
		
		read(handle, buf+3, seclen);
		seclen+=3;

		section=buf[0];
		sectionnum=buf[6];
		maxsec=buf[7];
	} while (loopc < maxsec*2 && ((section&mask[0]!=filter[0]) ||
		 pid!=PID || sectionnum!=secnum));
	msecnum=maxsec;
	CloseFilter(handle);
	return seclen;
}

int DVB::GetSection(uint8_t *buf, 
		    uint16_t PID, uint8_t TID, uint16_t TIDExt, 
		    uint16_t FilterTIDExt, 
		    uint8_t secnum, uint8_t &msecnum) 
{
	uint8_t filter[16], mask[16];
	
	memset(filter, 0, 16);
	memset(mask, 0, 16);
	
	filter[0]=TID;
	mask[0]=0xff;
	if (TIDExt!=0xffff) {
		filter[1]=(TIDExt>>8);
		filter[2]=TIDExt&0xff;
		mask[1]=(FilterTIDExt>>8);
		mask[2]=FilterTIDExt&0xff;
		}
	
	return GetSection(buf, PID, filter, mask, secnum, msecnum);
}		

void DVB::get_front(void) 
{
	set_vpid(0);
	set_apid(0);
	set_ttpid(0);
	set_pcrpid(0);
	voltage = SEC_VOLTAGE_13; 
	tone = SEC_TONE_OFF;
}	        
  

int DVB::check_frontend()
{
	int err;
	
   	if (( err = ioctl(fd_frontend, FE_READ_STATUS, &status))< 0 )
	    return err;
	if (( err = ioctl(fd_frontend, FE_READ_SIGNAL_STRENGTH, &signal))< 0) 
	    return err;
	if (( err = ioctl(fd_frontend, FE_READ_SNR, &snr))< 0) 
	    return err;
	if (( err = ioctl(fd_frontend, FE_READ_BER, &ber))< 0) 
	    return err;
	if (( err = ioctl(fd_frontend, FE_READ_UNCORRECTED_BLOCKS, 
			  &uncorrected_blocks))< 0) return err;
	return 0;
}


int DVB::tune_it(struct dvb_frontend_parameters *front_param)
{
	if (ioctl(fd_frontend, FE_SET_FRONTEND, front_param) <0){
		perror("setfront front");
		return -1;
	}
	return 0;
}


void DVB::set_front(void) 
{
	set_vpid(0);  
	set_apid(0);
	set_pcrpid(0);
	set_ttpid(0);

	if (front_type==FE_QPSK) {
		ioctl(fd_frontend, FE_SET_TONE, SEC_TONE_OFF);
		ioctl(fd_frontend, FE_SET_VOLTAGE, voltage);
		usleep(15000);
		if(ioctl(fd_frontend, FE_DISEQC_SEND_MASTER_CMD, &dcmd) < 0)
			perror("set_lnb");
		usleep(15000);
		if(ioctl(fd_frontend, FE_DISEQC_SEND_BURST, burst) < 0)
		  perror("set_lnb");
		usleep(15000);
		if(ioctl(fd_frontend, FE_SET_TONE, tone) < 0)
			perror("set_lnb");
		//usleep(70000);
	}
	tune_it(&front_param);
}	        

void DVB::set_diseqc_nb(int nr) 
{
	dcmd.msg[0]=0xe0;
	dcmd.msg[1]=0x10;
	dcmd.msg[2]=0x38;
	dcmd.msg[3]=0xF0 | 
		((nr * 4) & 0x0F) | 
		(tone ? 1 : 0) |
		(voltage ? 2 : 0);
	dcmd.msg[4]=0x00;
	dcmd.msg[5]=0x00;
	dcmd.msg_len=4;
	burst=(nr&1) ? SEC_MINI_B : SEC_MINI_A;
}

void DVB::set_ttpid(ushort ttpid) 
{  
	if (ttpid==0 || ttpid== NOPID || ttpid==0x1fff) {
		ioctl(fd_demuxtt, DMX_STOP, 0);
		return;
	}
	pesFilterParamsTT.pid     = ttpid;
	pesFilterParamsTT.input   = DMX_IN_FRONTEND; 
	pesFilterParamsTT.output  = DMX_OUT_DECODER; 
	pesFilterParamsTT.pes_type = DMX_PES_TELETEXT; 
	pesFilterParamsTT.flags   = DMX_IMMEDIATE_START;
	if (ioctl(fd_demuxtt, DMX_SET_PES_FILTER, 
		  &pesFilterParamsTT) < 0) {
	  printf("PID=%04x\n", ttpid);
		perror("set_ttpid");
	}
}

void DVB::set_vpid(ushort vpid) 
{  
	if (vpid==0 || vpid==NOPID || vpid==0x1fff) {
		ioctl(fd_demuxv, DMX_STOP, 0);
		return;
	}
	pesFilterParamsV.pid     = vpid;
	pesFilterParamsV.input   = DMX_IN_FRONTEND; 
	if (dvr_enabled)
		pesFilterParamsV.output  = DMX_OUT_TS_TAP; 
	else 
		pesFilterParamsV.output  = DMX_OUT_DECODER; 

	pesFilterParamsV.pes_type = DMX_PES_VIDEO;
	pesFilterParamsV.flags   = DMX_IMMEDIATE_START;
	if (ioctl(fd_demuxv, DMX_SET_PES_FILTER, 
		  &pesFilterParamsV) < 0)
		perror("set_vpid");
}

void DVB::set_apid(ushort apid) 
{  
	if (apid==0 || apid==NOPID || apid==0x1fff) {
		ioctl(fd_demuxa, DMX_STOP, apid);
		return;
	}

	pesFilterParamsA.pid = apid;
	pesFilterParamsA.input = DMX_IN_FRONTEND; 
	if (dvr_enabled)
		pesFilterParamsA.output = DMX_OUT_TS_TAP; 
	else
		pesFilterParamsA.output = DMX_OUT_DECODER;
 
	pesFilterParamsA.pes_type = DMX_PES_AUDIO; 
	pesFilterParamsA.flags = DMX_IMMEDIATE_START;
	if (ioctl(fd_demuxa, DMX_SET_PES_FILTER, 
		  &pesFilterParamsA) < 0)
		perror("set_apid");
}

void DVB::set_pcrpid(ushort pcrpid) 
{  
	if (pcrpid==0 || pcrpid==NOPID || pcrpid==0x1fff) {
		ioctl(fd_demuxpcr, DMX_STOP, pcrpid);
		return;
	}

	pesFilterParamsA.pid = pcrpid;
	pesFilterParamsA.input = DMX_IN_FRONTEND; 
	if (dvr_enabled)
		pesFilterParamsA.output = DMX_OUT_TS_TAP; 
	else
		pesFilterParamsA.output = DMX_OUT_DECODER;
 
	pesFilterParamsA.pes_type = DMX_PES_PCR; 
	pesFilterParamsA.flags = DMX_IMMEDIATE_START;
	if (ioctl(fd_demuxpcr, DMX_SET_PES_FILTER, 
		  &pesFilterParamsA) < 0)
		perror("set_pcrpid");
}



istream &operator>>(istream &ins, nokiaconv &x)
{
	int n=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	char dummy;
	int current_sat = -1;
	int current_tp = -1;
	int dint;

	enum { NSAT=0, NNET, NTRP, NCHAN, NEND};
	static char *nokiakeys[]={
		":SAT", ":NET", ":TRP", ":CHN", ":END", NULL
	};

	while(!ins.eof()){
		streampos pos = ins.tellg();
		ins >> keybuf;
		n=findkey(keybuf, nokiakeys);
		if (n<0) {
			ins.seekg(pos);
			break;
		}
		switch(n){
		case NSAT:
		{
			double did;
			int id=0;
			int lnbid = 5; 
			int found = 0;

			getname(sname,ins);
			//cerr << "Satellite \"" << sname << "\"" << endl;
			for(int i=0; i < x.lnb_sat.n; i++){
				if (!strcmp(x.lnb_sat.sat_names[i],sname)){
					lnbid = x.lnb_sat.diseqc[i]; 
					id = x.lnb_sat.satid[i]; 

					found = 1;
					break;
				}
					
			}

			x.dvb->AddLNB(lnbid, 1, 
				      9750000, 10600000, 
				      11700000, 
				      lnbid, 
				      NOID, NOID);
			
			ins >> did;
			current_sat =
				x.dvb->AddSat( id, lnbid,
					       sname, 
					       10700000 , 
					       12700000);
			ins >> dummy;
			
			break;
		}

		case NNET:
			getname(sname,ins);
			//cerr << "  Network \"" << sname << "\""<< endl; 
			ins >> dint;
			break;

		case NTRP:
		{
			Transponder trans;

			ins >> dec >> trans.id;
			ins >> trans.freq;
			ins >> trans.srate;
			ins >> dint;
			ins >> dummy;
			if (dummy == 'H') trans.pol = 1;
			if (dummy == 'V') trans.pol = 0;
			ins >> dint;

			trans.satid = x.dvb->sats[current_sat].id;
			trans.type = 1;
			trans.freq *= 10;
			trans.srate *= 100;

			ins >> dint;
			ins >> dummy;
			ins >> dint;

			switch (dint){
			case 2:
				trans.fec = FEC_1_2;
				break;
			case 3:
				trans.fec = FEC_2_3;
				break;
			case 4:
				trans.fec = FEC_3_4;
				break;
			case 6:
				trans.fec = FEC_5_6;
				break;
			case 8:
				trans.fec = FEC_7_8;
				break;
			}

			current_tp = x.dvb->AddTP(trans);
			//cerr << "    Transponder "<< trans.id  << endl;
			break;
		}

		case NCHAN:
		{
			Channel chan;
			int cnum;

			getname(sname,ins);
			strncpy(chan.name, sname, maxname);
			ins >> chan.pnr;
			ins >> dummy;
			if (dummy == 'T'){
				ins.ignore(20, ':');
				ins.seekg(ins.tellg()-streampos(1));

				chan.satid = x.dvb->sats[current_sat].id;
				chan.tpid = x.dvb->tps[current_tp].id;
				cnum = x.dvb->AddChannel(chan);

				//cerr << "      Channel "<< sname  
				//     << " (" << cnum << ")" << endl;
			} else 	{
				if (dummy == 'R'){
					ins.ignore(20, ':');
					ins.seekg(ins.tellg()-streampos(1));
				} else {
					ins.seekg(pos);
					ins.ignore(80,0x0a);
				}
			}
			break;
		}

		case NEND:
			//cerr << "ALL DONE" << endl;
			return ins;
		}		
	}
	return ins;
}

static int get_keylen(istream &ins, char *keybuf)
{
	streampos pos = ins.tellg();
	int klen = strlen(keybuf);
	if (klen>2 && keybuf[1]!= '/' &&
	    keybuf[0]=='<' && keybuf[klen-1]=='>'){
		keybuf[klen-2]='\0';
		klen--;
		ins.seekg(pos-streampos(2));
	}
	return klen;
}

static int find_xml_key(istream &ins, char *keybuf, char *keys[])
{
	char *s;
	int n;
	streampos pos = ins.tellg();
	ins >> keybuf;
	int klen = get_keylen(ins, keybuf);
	s=keybuf;
	while (s[0] != '=' && s != keybuf+klen)s++;
	s[0]=0;
	ins.seekg(pos + streampos((s-keybuf) +1) ); // go past =
	n=findkey(keybuf, keys);
	if (n<0) {
		ins.seekg(pos);
		cerr << "Unknown tag: " << keybuf << endl;
	}
	return n;
}

int xmlconv::read_sat(istream &ins, int csat)
{
	int n=-1;
	int lnbid=-1;
	int satid;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	enum { XSATN, XSLNB, XSATID, XTRANS, XSATEND, XEND, XNEND};
	static char *xsat[]={
		"name", "lnb", "id", "<transponder", "</satellite>",
		">", "/>", NULL
	};
	while(!ins.eof()){
		if ( (n = find_xml_key( ins, keybuf, xsat)) < 0) break;
		switch(n){
		case XSATN:
			getname(sname,ins);			
			break;

		case XSLNB:
			ins >> satid;
			break;

		case XTRANS:
			if (csat >= 0)
				read_trans(ins, csat);
			else
				return -1;
			break;

		case XSATID:
			ins >> satid;
			break;

		case XSATEND:
			return 0;
			break;

		case XEND:
			if (satid >=0 && lnbid >= 0)
				csat =	dvb->AddSat(satid, 
						    lnbid,
						    sname, 
						    10700000 , 
						    12700000);
			break;

		case XNEND:
			return 0;
			break;

		default:
			skip_tag(ins,keybuf);
			break;
		}
	}

	return 0;
}

int xmlconv::read_trans(istream &ins, int csat)
{
	int n=-1;
	int ctp=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	enum { XTYPE=0, XFREQ, XSRATE, XPOL, XFEC, XSERV, 
	       XTRANSEND, XEND, XNEND };
	static char *xtrans[]={
		"type", "freq", "srate", "polarity", "fec", 
		"<service", "</transponder>", 
		">", "/>", NULL
	};
	Transponder trans;
	trans.satid = dvb->sats[csat].id;
	trans.fec = FEC_AUTO;
	trans.id = NOID;

	while(!ins.eof()){
		if ( (n = find_xml_key( ins, keybuf, xtrans)) < 0) break;
		switch(n){
		case XTYPE:
			getname(sname,ins);
			switch(sname[0]){
			case 'S':
				trans.type = FE_QPSK;
				break;
			case 'T':
				trans.type = FE_OFDM;
				break;
			case 'C':
				trans.type = FE_QAM;
				break;
			}
			break;

		case XFREQ:
			getname(sname,ins);
			trans.freq=atoi(sname);
			break;

		case XSRATE:
			getname(sname,ins);
			trans.srate=atoi(sname);
			break;

		case XPOL:
			getname(sname,ins);
			if (sname[0] == 'H') trans.pol = 1;
			if (sname[0] == 'V') trans.pol = 0;
			break;

		case XFEC:
			int dint;
			getname(sname,ins);
			dint = atoi(sname);

			switch (dint){
			case 2:
				trans.fec = FEC_1_2;
				break;
			case 3:
				trans.fec = FEC_2_3;
				break;
			case 4:
				trans.fec = FEC_3_4;
				break;
			case 6:
				trans.fec = FEC_5_6;
				break;
			case 8:
				trans.fec = FEC_7_8;
				break;
			}

			break;


		case XSERV:
			if (ctp>=0)
				read_serv(ins,ctp,csat);
			break;

		case XTRANSEND:
			return 0;
			break;

		case XEND:
			ctp = dvb->AddTP(trans);
			break;

		case XNEND:
			return 0;
			break;
		default:
			skip_tag(ins,keybuf);
			break;

		}
	}

	return 0;
}

int xmlconv::read_serv(istream &ins, int ctp, int csat)
{
	int n=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	enum { XSID=0, XCA, XDESC, XSTREAM, 
	       XSERVEND, XEND, XNEND };
	static char *xserv[]={
		"id", "ca", 
		"<description", 
		"<stream", "</service>",  		
		">", "/>", 
		"<ca_descriptor", "<descriptor",
		"<country_availability", "<ca_system_id", "<time_shifted_copy_of",
		NULL
	};
	Channel chan;
	int nchan=-1;

	chan.satid = dvb->sats[csat].id;
	chan.tpid = dvb->tps[ctp].id;

	while(!ins.eof()){
		if ( (n = find_xml_key( ins, keybuf, xserv)) < 0) break;
		switch(n){
		case XSID:
			getname(sname,ins);
			chan.pnr = atoi(sname);
			nchan = dvb->AddChannel(chan);
			break;

		case XCA:
			getname(sname,ins);
			if (nchan >= 0)
				dvb->chans[nchan].type = atoi(sname);
			else
				chan.type = atoi(sname);
			break;

		case XDESC:
			if (nchan>=0)
				read_desc(ins, nchan);
			else
				return -1;
			break;

		case XSTREAM: 
			if (nchan>=0)
				read_stream(ins,nchan);
			else
				return -1;
			break;

		case XSERVEND: 
			return 0;
			break;

		case XEND: 
			break;

		case XNEND:
			return 0;
			break;
		default:
			skip_tag(ins,keybuf);
			break;
		}
	}

	return 0;
}

int xmlconv::read_desc(istream &ins, int nchan)
{
	int n=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	enum { XTAG=0, XTYPE, XPROV, XSNAME, XDESCEND, 
	       XEND, XNEND};
	static char *xdesc[]={
		"tag", "type", "provider_name", "service_name", 
		"</description>", ">", "/>", NULL
	};

	while(!ins.eof()){
		if ( (n = find_xml_key( ins, keybuf, xdesc)) < 0) break;
		switch(n){
		case XTAG:
			getname(sname,ins);
			break;

		case XTYPE:
			getname(sname,ins);
			break;

		case XPROV:
			getname(sname,ins);
			break;

		case XSNAME:
			getname(sname,ins);
			dvb2txt(dvb->chans[nchan].name,sname,MAXNAM);
			break;

		case XDESCEND:
			return 0;
			break;

		case XEND:
			break;

		case XNEND:
			return 0;
			break;

		default:
			skip_tag(ins,keybuf);
			break;
		}
	}

	return 0;
}

int xmlconv::read_stream(istream &ins, int nchan)
{
	int n=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	int type = -1;
	int apids = 0;
	uint16_t pid = NOPID;
	enum { XTYPE, XPID, XISO, XSTREAMEND, XEND, XNEND };
	static char *xstream[]={
		"type", "pid", "<iso_639",
		"</stream>", ">", "/>", 
		"<ca_descriptor","<descriptor","<teletext","<stream_id", 
		"<canal_radio", "<audio_info", "<description", "<ac3_descriptor",
		NULL
	};

	while(!ins.eof()){
		if ( (n = find_xml_key( ins, keybuf, xstream)) < 0) break;
		switch(n){
		case XTYPE:
			getname(sname,ins);
			type = atoi(sname);
			break;

		case XPID:
			getname(sname,ins);
			pid = atoi(sname);
			switch(type){
			case 1:
			case 2:
				if (pid != NOPID)
					dvb->chans[nchan].vpid = pid;
				break;
			case 3:
			case 4:
				if (pid == NOPID) break;
				apids = dvb->chans[nchan].apidnum;
				if (apids >= MAXAPIDS) break;
				dvb->chans[nchan].apidnum++;
				dvb->chans[nchan].apids[apids]=pid;
				break;
			case 6:
				if (pid != NOPID)
					dvb->chans[nchan].ttpid = pid;
				break;				
			}
			break;

		case XSTREAMEND:
			return 0;
			break;

		case XEND:
			break;
			
		case XNEND:
			return 0;
			break;

		case XISO:
			read_iso639(ins, nchan, apids);
			break;

		default:
			skip_tag(ins,keybuf);
			break;
		}
	}

	return 0;
}

int xmlconv::read_iso639(istream &ins, int nchan, int apids)
{
	int n=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	enum { XTYPE, XLAN,XISOEND, XEND, XNEND };
	static char *xiso[]={
		"type", "language", "</iso_639>",
		">", "/>", NULL
	};

	while(!ins.eof()){
		if ( (n = find_xml_key( ins, keybuf, xiso)) < 0) break;
		switch(n){
		case XTYPE:
			getname(sname,ins);
			break;

		case XLAN:
			getname(sname,ins);
			strncpy(dvb->chans[nchan].apids_name+apids*4, sname, 4);
			break;

		case XISOEND:
			return 0;
			break;

		case XEND:
			break;
			
		case XNEND:
			return 0;
			break;

		default:
			skip_tag(ins,keybuf);
			break;
		}
	}

	return 0;
}

int xmlconv::skip_tag(istream &ins, char *tag)
{
	char sname[MAXNAM*2];
	char endtag[MAXNAM];
	int found = 0;
	streampos pos = ins.tellg();
	
	ostrstream etag(endtag,MAXNAM);
	etag << "</" << tag+1 << ">" << ends;
	int tlen = strlen(endtag)-1;
	
//	cerr << "find: " << endtag << endl;
	ins >> sname;
	if (sname[0] == '>')
		while(!found){
			if (!strncmp(sname,endtag,tlen)) 
				found=1;
			else 
				ins >> sname;
		}
	else {
		ins.seekg(pos);
		ins.ignore(1000,'>');
		pos = ins.tellg();
		ins.seekg(pos-streampos(2));
		ins >> sname;
		if (sname[0] == '/')
			ins.seekg(pos);
		else 
			while(!found){
				if (!strncmp(sname,endtag,tlen)) 
					found=1;
				else
					ins >> sname;
			}

	}
	return 0;
}

istream &operator>>(istream &ins, xmlconv &x)
{
	int n=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	int current_sat = -1;
	int nsat = 0;

	enum { XMLSTART=0, XSAT, XNSAT, XLNB, XEND, XNEND};
	static char *xmltags[]={
		"<?xml","<satellite", "<satellite>",
		"<lnb",
		">", "/>", NULL};

	while(!ins.eof()){
		streampos pos = ins.tellg();
		ins >> keybuf;
		n=findkey(keybuf, xmltags);
		if (n<0) {
			ins.seekg(pos);
			cerr << "Unknown tag: " << keybuf << endl;
			break;
		}
		switch(n){
		case XMLSTART:
			cerr << "xml start found" << endl;
			ins.ignore(80,'>');
			break;

		case XNSAT:
		{
			int clnb;
			int lnbid = 5; 
			int satid = -1;

			if (nsat > XML_MAX_SAT) break;
			strcpy(sname,x.lnb_sat.sat_names[nsat]);
			lnbid = x.lnb_sat.diseqc[nsat]; 
			clnb = x.dvb->AddLNB(lnbid, 1, 
				      9750000, 10600000, 
				      11700000, 
				      lnbid, 
				      NOID, NOID);
			
			satid = x.lnb_sat.satid[nsat];
			current_sat =
				x.dvb->AddSat(satid, 
					       lnbid,
					       sname, 
					       10700000 , 
					       12700000);
			nsat++;
			x.read_sat(ins, current_sat);
			break;
		}

		case XSAT:
		{
			cerr << "no sat name" << endl;
			x.read_sat(ins, -1);
			break;
		}

		case XLNB:

			break;

		default:
			x.skip_tag(ins,keybuf);
			break;
		}		
	}
	return ins;
}

int get_dvbrc(char *path, DVB &dv, int dev, int len)
{
        ifstream dvbin;

	dvbin.open(path);
	if (!dvbin){
		cerr << "Using default dvbrc." << endl;
		const char *home = getenv("HOME");
		const char *file = ".dvbrc";
		ostrstream str(path,len-1);

		str << home << "/" << file ;
		if (dev)
			str << "." << dev ;
		str << ends;

		dvbin.open(path);
	}
        if (dvbin) {
                dvbin >> dv;
                return 1;
        }

        return 0;
}

int set_dvbrc(char *path, DVB &dv, int dev, int len)
{
        ofstream dvbout;

	dvbout.open(path);
	if (!dvbout){
		cerr << "Using default dvbrc." << endl;
		const char *home = getenv("HOME");
		const char *file = ".dvbrc";
		ostrstream str(path,len-1);

		str << home << "/" << file ;
		if (dev)
			str << "." << dev ;
		str << ends;

		dvbout.open(path);
	}
        if (dvbout) {
                dvbout << dv;
                return 1;
        }

        return 0;
}



