/* 
    GTVscreen Widget for v4l frame grabber driver

    Copyright (C) 1999 Marcus Metzler (mocm@metzlerbros.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * gtvscreen.c 
 */

//#define DEBUG 1
#include <gtvscreen2.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <sys/wait.h>


#ifdef HAVE_LIBXXF86VM
#include <X11/extensions/xf86vmode.h>
#endif

#ifdef HAVE_LIBXV
#include <X11/extensions/Xv.h>
#include <X11/extensions/Xvlib.h>
#endif
#include "xv.h"




static int get_clip();
static void gtvscreen_realize();
static void gtvscreen_init();
guint gtvscreen_get_type();
static void gtk_gtvscreen_size_allocate();
static void gtvscreen_destroy();
static void set_capture();
static void set_geo();
static void setmaxsize();
static void gtvscreen_send_configure();
static gint paint();
static void grab_image();
static unsigned int which_format();
static void gtvscreen_all_new();

#ifdef HAVE_LIBXXF86VM
static int toggle_fullscreen();
#endif

static GdkFilterReturn pConfigFilter();
static gint pConfig();

static XErrorHandler xerror(); 

static GtkWidgetClass *parent_class = NULL;

static const gint MAXX = 1024;
static const gint MAXY = 768;

#define TIMEOUT 150

enum {
        CHINPUT,
        CHCAPTURE,
        CHOVERLAY,
        CHATTRIB,
        LAST_SIGNAL
};

static guint gtvscreen_signals[LAST_SIGNAL] = { 0 };

static int myioctl(int fd, int call, void *pointer)
{
	int status=0;

#ifndef NOCTL
	status = ioctl(fd, call, pointer);
#endif
	return status;
}

XErrorHandler xerror(Display *dpy, XErrorEvent *event)
{
  switch (event->error_code){
  case BadWindow:
#ifdef DEBUG
    printf(_("Bad Window ingnored\n"));
#endif
    return NULL;
    break;

  case BadMatch:
#ifdef DEBUG
    printf("BadMatch\n");
#endif
    return NULL;
    break;
 
  case BadColor:
#ifdef DEBUG
    printf("BadColor\n");
#endif
    return NULL;
    break;
    
    default:
      return NULL;
  }
}                    

static void gtvscreen_class_init(GtvscreenClass * klass)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = (GtkObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;

	parent_class = gtk_type_class(gtk_widget_get_type());


	gtvscreen_signals[CHINPUT] =
	    gtk_signal_new("chinput",
			   GTK_RUN_FIRST,
			   object_class->type,
			   GTK_SIGNAL_OFFSET(GtvscreenClass, chinput),
			   gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

	gtvscreen_signals[CHCAPTURE] =
	    gtk_signal_new("chcapture",
			   GTK_RUN_FIRST,
			   object_class->type,
			   GTK_SIGNAL_OFFSET(GtvscreenClass, chcapture),
			   gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

	gtvscreen_signals[CHOVERLAY] =
	    gtk_signal_new("choverlay",
			   GTK_RUN_FIRST,
			   object_class->type,
			   GTK_SIGNAL_OFFSET(GtvscreenClass, choverlay),
			   gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

	gtvscreen_signals[CHATTRIB] =
	    gtk_signal_new("chattrib",
			   GTK_RUN_FIRST,
			   object_class->type,
			   GTK_SIGNAL_OFFSET(GtvscreenClass, chattrib),
			   gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);



	gtk_object_class_add_signals(object_class, gtvscreen_signals,
				     LAST_SIGNAL);

	object_class->destroy = gtvscreen_destroy;
	widget_class->size_allocate = gtk_gtvscreen_size_allocate;
	widget_class->realize = gtvscreen_realize;

	klass->chinput   = NULL;	// I dont need to know yet.
	klass->chcapture = NULL;
	klass->choverlay = NULL;
	klass->chattrib  = NULL;
}



static void gtk_gtvscreen_size_allocate(GtkWidget * widget,
					GtkAllocation * allocation)
{
	g_return_if_fail(widget != NULL);
	g_return_if_fail(IS_GTVSCREEN(widget));
	g_return_if_fail(allocation != NULL);

	widget->allocation = *allocation;

	if (GTK_WIDGET_REALIZED(widget)) {
		gdk_window_move_resize(widget->window,
				       allocation->x, allocation->y,
				       allocation->width,
				       allocation->height);

		gtvscreen_send_configure(GTVSCREEN(widget));
	}
}

void gtk_gtvscreen_size(Gtvscreen * gtv, gint width, gint height)
{
	g_return_if_fail(gtv != NULL);
	g_return_if_fail(IS_GTVSCREEN(gtv));

	GTK_WIDGET(gtv)->requisition.width = width;
	GTK_WIDGET(gtv)->requisition.height = height;
}


gint refresh_timer(gpointer data)
{
	Gtvscreen *gtv;

#ifndef NOCARD
	int scr, depth;
	XSetWindowAttributes xswa;
	unsigned long mask;
	Window rwin, rootW;
	Display *disp;
	int x, y, width, height;

	gtv = (Gtvscreen *) data;

	x = gtv->vwin.x;
	y = gtv->vwin.y;

	width = gtv->vwin.width;
	height = gtv->vwin.height;


	if (!gtv->OVL_ON)
		return 0;
#ifdef DEBUG
	printf("REFRESH\n");
#endif
	disp = GDK_DISPLAY();
	scr = DefaultScreen(disp);
	rootW = RootWindow(disp, scr);
	depth = gtv->vbuf.depth;

	xswa.override_redirect = TRUE;
	xswa.backing_store = NotUseful;
	xswa.save_under = FALSE;
	xswa.background_pixel = BlackPixel(disp, scr);
	xswa.backing_store = NotUseful;
	mask = (CWBackPixel | CWSaveUnder | CWBackingStore |
		CWOverrideRedirect);
	if (gtv->Move) {
		rwin = XCreateWindow(disp, rootW, 0, 0,
				     gtv->vbuf.width, gtv->vbuf.height, 0,
				     CopyFromParent, InputOutput,
				     CopyFromParent, mask, &xswa);
		gtv->Move = FALSE;
	} else {
		rwin = XCreateWindow(disp, rootW,
				     x - 20, y - 20,
				     width + 20, height + 20,
				     0, CopyFromParent, InputOutput,
				     CopyFromParent, mask, &xswa);
	}
	gtv->lastrwin = rwin;
	XMapWindow(disp, rwin);
	XUnmapWindow(disp, rwin);
	XDestroyWindow(disp, rwin);
#endif
	gtv->refresh = 1;

	return 0;
}

static void refresh(gpointer data)
{
	Gtvscreen *gtv;
	
	gtv = (Gtvscreen *) data;
	
	if (gtv->remote == -1) return;

	if (gtv->refresh || !gtv->Move){
		gtv->refresh = 0;
		gtv->timeout = 	gtk_timeout_add(
			TIMEOUT,refresh_timer,(gpointer)gtv);
	}
}

guint gtvscreen_get_type()
{
	static guint gtv_type = 0;

	if (!gtv_type) {
		GtkTypeInfo gtv_info = {
			"Gtvscreen",
			sizeof(Gtvscreen),
			sizeof(GtvscreenClass),
			(GtkClassInitFunc) gtvscreen_class_init,
			(GtkObjectInitFunc) gtvscreen_init,
			(GtkArgGetFunc) NULL,
			(GtkArgSetFunc) NULL,
		};

		gtv_type = gtk_type_unique(gtk_vbox_get_type(), &gtv_info);
	}

	return gtv_type;
}

GtkWidget *gtvscreen_new()
{
	return gtvscreen_new_with_dev((gchar *) NULL);
}

GtkWidget *gtvscreen_new_with_dev(gchar *devname)
{
	GtkWidget *gtvw;
	Gtvscreen *gtv;

	gtvw = GTK_WIDGET(gtk_type_new(gtvscreen_get_type()));
	gtv = GTVSCREEN(gtvw);

	if (devname) {
		printf(_("setting dev to %s\n"), devname);
		strcpy(gtv->devname, devname);
	} else {
		printf(_("setting dev to %s\n"), "/dev/video");
		strcpy(gtv->devname, "/dev/video");
	}

	if ((gtv->fbttv = open(gtv->devname, O_RDWR)) < 0) {
		fprintf(stderr, "Cannot open %s\n", gtv->devname);
		exit(1);
	}

	gtvscreen_all_new(gtv);

	return gtvw;
}


GtkWidget *gtvscreen_new_with_fd(int fd)
{
	GtkWidget *gtvw;
	Gtvscreen *gtv;

	gtvw = GTK_WIDGET(gtk_type_new(gtvscreen_get_type()));
	gtv = GTVSCREEN(gtvw);
	gtv->fbttv = fd;
	printf(_("setting file descriptor for device to %d\n"), gtv->fbttv);
	gtvscreen_all_new(gtv);
	return gtvw;
}

static void gtvscreen_all_new(Gtvscreen *gtv){
	unsigned char *tm;
	int msize;

#ifndef NOCARD
	if (myioctl(gtv->fbttv, VIDIOCGCAP, &(gtv->vcap)))
		perror("get capabilities");

	if (!(gtv->vcap.type & VID_TYPE_OVERLAY)){
		if (gtv->remote != -1) gtv->remote = 1;
		fprintf(stderr,"no overlay possible\n");
	}

	if (!(gtv->vcap.type & VID_TYPE_CAPTURE)){
		gtv->remote = -1;
		gtv->OVL_ON = 1;
		fprintf(stderr,"no capture possible\n");
	}


	if (myioctl(gtv->fbttv, VIDIOCGWIN, &(gtv->vwin)))
		perror("get window");
	if (myioctl(gtv->fbttv, VIDIOCGFBUF, &(gtv->vbuf)))
		perror("get frame buffer");
	/* Alloc memory for snapshot */

	if (gtv->remote != -1){
		if (myioctl(gtv->fbttv, VIDIOCGMBUF, &(gtv->mbuf))){
			perror("get mbuf");
			gtv->remote = -1;
		} else {
			msize = gtv->mbuf.size;
			fprintf(stderr, "msize: 0x%08x\n", msize);
			tm = (unsigned char *) mmap(0, msize, PROT_READ | 
						    PROT_WRITE,
						    MAP_SHARED, gtv->fbttv, 0);
			if (tm == (unsigned char *) -1) {
				fprintf(stderr,_("Cannot mmap memory for snapshot\n"));
				exit(1);
			}
#ifdef DEBUG
			fprintf(stderr, "tm: 0x%08x\n", (int) tm);
#endif
			gtv->TargetMem = tm;
		}
	}
#endif

}



#ifndef NOCARD
static Boolean onscreen(Gtvscreen * gtv, XWindowAttributes wts)
{
	int x1, x2, y1, y2;
	int wx1, wx2, wy1, wy2;
	GtkWidget *widget;
	Boolean xlap, ylap;

	widget = GTK_WIDGET(gtv);

	if (!(wts.map_state & IsViewable))
		return FALSE;

	x1 = gtv->vwin.x;
	y1 = gtv->vwin.y;
	x2 = x1 + gtv->vwin.width;
	y2 = y1 + gtv->vwin.height;

	wx1 = wts.x;
	wy1 = wts.y;
	wx2 = wx1 + wts.width + 2 * wts.border_width - 1;
	wy2 = wy1 + wts.height + 2 * wts.border_width - 1;

	xlap = ((wx1 < x1 && wx2 > x1) || (wx1 > x1 && wx1 < x2));
	ylap = ((wy1 < y1 && wy2 > y1) || (wy1 > y1 && wy1 < y2));

	if (xlap && ylap)
		return TRUE;
	else
		return FALSE;

}

#endif

static void gtvscreen_send_configure(Gtvscreen * gtv)
{
	GtkWidget *widget;
	GdkEventConfigure event;

	g_return_if_fail(gtv != NULL);
	g_return_if_fail(IS_GTVSCREEN(gtv));


	widget = GTK_WIDGET(gtv);

	event.type = GDK_CONFIGURE;
	event.window = widget->window;
	event.send_event = TRUE;
	event.x = widget->allocation.x;
	event.y = widget->allocation.y;
	event.width = widget->allocation.width;
	event.height = widget->allocation.height;

	gtk_widget_event(widget, (GdkEvent *) & event);
	set_geo(gtv);
}

/* Realize */
static void gtvscreen_realize(GtkWidget * widget)
{
	Gtvscreen *gtv;
	GdkWindowAttr attributes;
	gint attributes_mask;
	uint rwidth;
	GdkWindow *win;
	Display *disp;
	int major, minor, bank, ram;
	GtkWidget *parent;
	GdkColor keycolor;
	GdkCursor *ptr;

	g_return_if_fail(widget != NULL);
	g_return_if_fail(IS_GTVSCREEN(widget));


	GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);
	gtv = GTVSCREEN(widget);

	attributes.x = widget->allocation.x;
	attributes.y = widget->allocation.y;
	attributes.width = widget->allocation.width;
	attributes.height = widget->allocation.height;
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.event_mask = gtk_widget_get_events(widget) |
	    GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK |
	    GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK |
	    GDK_POINTER_MOTION_HINT_MASK;
	attributes.visual = gtk_widget_get_visual(widget);
	gtv->visual = attributes.visual;

	attributes.colormap = gtk_widget_get_colormap(widget);
	keycolor.red = 0;
        keycolor.green = 10280;
        keycolor.blue = 0;

        // Allocate color
        gdk_color_alloc (attributes.colormap, &keycolor);
    

	attributes_mask =
	    GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
	widget->window =
	    gdk_window_new(widget->parent->window, &attributes,
			   attributes_mask);
	ptr = gdk_cursor_new(GDK_RTL_LOGO);
	if(ptr){
		gdk_window_set_cursor(widget->window, ptr);
		gdk_cursor_destroy(ptr);
	}
	widget->style = gtk_style_attach(widget->style, widget->window);

	gdk_window_set_user_data(widget->window, widget);
	gtk_style_set_background(widget->style, widget->window,
				 GTK_STATE_SELECTED);

	disp = (Display *) GDK_DISPLAY();
	win = widget->window;

	gtk_widget_set_usize(GTK_WIDGET(gtv), 0, 0);
	gtv->vwin.width = attributes.width;
	gtv->vwin.height = attributes.height;
	gtv->vwin.x = 0;
	gtv->vwin.y = 0;
	gtv->vbuf.width = gdk_screen_width();
	gtv->vbuf.height = gdk_screen_height();
	gtv->vbuf.base = 0;
	gtv->vwin.clipcount = 0;

	rwidth = gtv->vbuf.width;

#ifdef DEBUG
	fprintf(stderr, _("Screensize:%dx%d\n"), gdk_screen_width(),
		gdk_screen_height());
#endif

#ifndef NOCARD
#ifdef HAVE_LIBXXF86DGA
	if (gtv->remote != 1 && XF86DGAQueryVersion(disp, &major, &minor)) {
#ifdef DEBUG
		fprintf(stderr, "DGA Version %d.%d\n", major, minor);
#endif
		switch (major) {
		case 1:
			XF86DGAGetVideoLL(disp, DefaultScreen(disp),
					  (void *) &(gtv->vbuf.base),
					  &rwidth, &bank, &ram);
			break;
		case 2:
			XF86DGAGetVideoLL(disp, DefaultScreen(disp),
					  (void *) &(gtv->vbuf.base),
					  &rwidth, &bank, &ram);
			break;

		default:
			gtv->remote = 1;
		}
	} else {
#ifdef DEBUG
		fprintf(stderr, _("Can`t use DGA\n"));
#endif
		if (!gtv->remote) gtv->remote = 1;
	}
	//fprintf(stderr,"addr: 0x%08x\n",gtv->vbuf.base);


#endif
	if (gtv->realw)
		rwidth = gtv->realw;

	if (gtv->bpp)
		gtv->vbuf.depth = gtv->bpp;
	else {
		Screen *s;
 		int n,i;
		Display *dpy;
		XPixmapFormatValues *pf;

		
		dpy=XOpenDisplay(NULL);
		s = ScreenOfDisplay (dpy,DefaultScreen (dpy));


		gtv->vbuf.depth = DisplayPlanes (dpy, 0);
		pf = XListPixmapFormats(dpy,&n);
		i = 0;
		while (i < n) 
		{
			if ( pf[i].depth == gtv->vbuf.depth ){
				gtv->bpp = pf[i].bits_per_pixel;
				break;
			}
			i++;
		}
		
		fprintf(stderr,"Depth: %d\n",gtv->vbuf.depth);
		fprintf(stderr,"BPP: %d\n",gtv->bpp);

		XFree(pf);
		XCloseDisplay(dpy);

	}

	gtv->Format = which_format(gtv->bpp);

	gtv->vbuf.bytesperline =
	    rwidth * (0x38 & (gtv->vbuf.depth + 1)) / 8;
	gtv->vp.depth = gtv->vbuf.depth;
	gtv->vp.palette = which_format(gtv->vbuf.depth);


	if (gtv->remote != 1){
		if (myioctl(gtv->fbttv, VIDIOCSFBUF, &gtv->vbuf)) {
			perror("set frame buffer");
			exit(1);
		}
	}
	setuid(getuid());
	myioctl(gtv->fbttv, VIDIOCSPICT, &gtv->vp);

#endif

	parent = gtk_widget_get_toplevel (widget);

	gtk_signal_connect_object_after(GTK_OBJECT(parent), 
					"map_event", (GtkSignalFunc)
					pConfig, GTK_OBJECT(widget));

	gtk_signal_connect_object_after(GTK_OBJECT(parent), "unmap_event",
			   (GtkSignalFunc) pConfig, GTK_OBJECT(widget));

	gtk_signal_connect_object_after(GTK_OBJECT(parent), "configure_event",
			   (GtkSignalFunc) pConfig, GTK_OBJECT(widget));

	XSelectInput(disp, DefaultRootWindow(disp), VisibilityChangeMask |
		     SubstructureNotifyMask | ExposureMask |
		     StructureNotifyMask);

	gdk_window_add_filter((GdkWindow *) & gdk_root_parent,
			      (GdkFilterFunc) pConfigFilter, gtv);

	setmaxsize(gtv);

	if (!gtv->image) {
		gtv->image =
		    gdk_image_new(GDK_IMAGE_FASTEST, gtv->visual, MAXX,
				  MAXY+256);
		gtv->gc = gdk_gc_new(widget->window);
	}

	gdk_window_show(widget->window);
	pConfig(gtv, 0, gtv);

	set_geo(gtv);
	XSetErrorHandler((XErrorHandler) xerror); 
}


static void gtvscreen_init(Gtvscreen * gtv)
{
	GdkColor *my;
	GdkColormap *cmap;
	char buf[16];
	ushort i;
	GdkVisual *v;
	GtkWidget *widget = GTK_WIDGET(gtv);

	g_return_if_fail(gtv != NULL);
	g_return_if_fail(IS_GTVSCREEN(gtv));


	GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(gtv), GTK_NO_WINDOW);
	gtv->gc = (GdkGC *) NULL;
	gtv->image = (GdkImage *) NULL;
	gtv->CMSET = FALSE;

	v = gtk_widget_get_visual(widget);
	if (v->type == GDK_VISUAL_PSEUDO_COLOR) {
		fprintf(stderr,
			_("Setting colormap for PseudoColor Visual\n"));
		my = gtv->colors;
		cmap = gdk_colormap_new(gdk_visual_get_best(), 256);
		for (i = 0; i < 16; i++) {
			gdk_color_black(cmap, &my[i]);
		}
		for (i = 0; i < 225; i++) {
			int r, rr;
			r = i % 25;
			rr = r % 5;
			sprintf(buf, "#%02X%02X%02X",
				(65535 * (r / 5) / 4),
				(65535 * (i / 25) / 8), (65535 * rr / 4));
			gdk_color_parse(buf, &my[i + 16]);
			gdk_color_alloc(cmap, &my[i + 16]);
		}
		gtk_widget_push_colormap(cmap);
		gtv->cmap = cmap;
		gtk_widget_set_colormap(widget, cmap);
	}
	// get picture settings and set reasonable values
	myioctl(gtv->fbttv, VIDIOCGPICT, &gtv->vp);

	gtv->vp.brightness = 32768;
	gtv->vp.colour = 32768;
	gtv->vp.hue = 32768;
	gtv->vp.contrast = 32768;
	gtv->vp.whiteness = 32768;
	gtv->vp.depth = gtv->bpp;
	gtv->vp.palette = gtv->Format;

	myioctl(gtv->fbttv, VIDIOCSPICT, &gtv->vp);


	// get input settings 
	myioctl(gtv->fbttv, VIDIOCGCAP, &gtv->vc);

	gtv->vc.norm = 0;
	gtv->vc.channel = 0;

	// get window settings 
	myioctl(gtv->fbttv, VIDIOCGWIN, &gtv->vwin);

	gtv->accel_group = gtk_accel_group_new();
	gtk_accel_group_attach(gtv->accel_group, GTK_OBJECT(gtv));


	// get audio settings (and set reasonable values)
	myioctl(gtv->fbttv, VIDIOCGAUDIO, &gtv->va);
	myioctl(gtv->fbttv, VIDIOCSAUDIO, &gtv->va);

	gtv->refresh = 1;
	gtv->Move = FALSE;

}


static void gtvscreen_destroy(GtkObject * object)
{
	Gtvscreen *gtv;

	g_return_if_fail(object != NULL);
	g_return_if_fail(IS_GTVSCREEN(object));

	gtv = GTVSCREEN(object);

	gtv->CAP_ON = 0;
	set_capture(gtv, 0);
	close(gtv->fbttv);
#ifdef DEBUG
	printf("Widget destroyed\n");
#endif

}


static void set_capture(Gtvscreen * gtv, int W)
{
	static int timer;
	int one = 1;
	int zero = 0;

	g_return_if_fail(gtv != NULL);
	g_return_if_fail(IS_GTVSCREEN(gtv));

	if (gtv->remote == 1 && gtv->OVL_ON) {
		gtv->OVL_ON = FALSE;
	}
#ifndef NOCARD

	if (GTK_WIDGET_REALIZED(gtv)) {

		switch (W) {
		case 1:
			if (gtv->CAP_ON) {
				if (gtv->OVL_ON) {
					get_clip(gtv);
					set_geo(gtv);
					if (myioctl
					    (gtv->fbttv, VIDIOCCAPTURE,
					     &one)) perror("capture on");
				} else {
					if (gtv->remote != -1){
						set_geo(gtv);
						myioctl(gtv->fbttv, 
							VIDIOCCAPTURE,
							&zero);
						gtv->start = TRUE;
						gtv->stop = FALSE;
						timer =
							gtk_idle_add(paint,
								     (gpointer)
								     gtv);
					}
				}
			}
			break;

		case 0:
		default:
			if (gtv->remote != -1){
				if (timer) {
					gtk_idle_remove(timer);
					gtv->stop = TRUE;
					grab_image(gtv);
					gtv->stop = FALSE;
					timer = 0;
				} else {
					myioctl(gtv->fbttv, VIDIOCCAPTURE,
						&zero);
				}
			}

		}
#endif
	}
}

#ifndef NOCARD
static void add_clip(struct video_clip *cr, int *ncr, uint x, uint y,
		    uint w, uint h)
{

	if (*ncr > MAXCLIP) {
		fprintf(stderr, _("Too many clipped windows\n"));
		return;
	}
	cr[*ncr].x = x;
	cr[*ncr].y = y;
	cr[*ncr].width = w;
	cr[*ncr].height = h;
	*ncr = *ncr + 1;
}

#endif

static void maketab(Gtvscreen * gtv, int ncr)
{
	g_return_if_fail(gtv != NULL);
	g_return_if_fail(IS_GTVSCREEN(gtv));

#ifndef NOCARD
	gtv->vwin.clips = gtv->cliprecs;
	gtv->vwin.clipcount = ncr;
	if (ncr && gtv->OVL_ON && gtv->CAP_ON && gtv->vwin.width
	    && gtv->vwin.height) {
		if (myioctl(gtv->fbttv, VIDIOCSWIN, &(gtv->vwin)))
			perror("set clipping");
	}
#endif
}

#ifndef NOCARD
static int get_clip(Gtvscreen * gtv)
{
	int ncr = 0;
	int x, y, x2, y2;
	Display *disp;
	XWindowAttributes wts;
	Window parent, win, parent2, root, rroot, *children, swin;
	uint nchildren, i;
	GtkWidget *widpar;

	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(gtv))
	    || !gtv->OVL_ON)
		return 0;


	disp = (Display *) GDK_DISPLAY();

	widpar = GTK_WIDGET(gtv);
	while (widpar->parent)
		widpar = widpar->parent;
	if (!(win = GDK_WINDOW_XWINDOW(widpar->window)))
		return 0;

	XQueryTree(disp, win, &rroot, &parent, &children, &nchildren);
	if (nchildren)
		XFree((char *) children);

	root = DefaultRootWindow(disp);
	swin = win;
	while (parent != root) {
		swin = parent;
		XQueryTree(disp, swin, &rroot, &parent, &children,
			   &nchildren);
		if (nchildren)
			XFree((char *) children);
	}
	XQueryTree(disp, root, &rroot, &parent2, &children, &nchildren);
	for (i = 0; i < nchildren; i++)
		if (children[i] == swin)
			break;
	i++;
	for (; i < nchildren; i++) {

		XGetWindowAttributes(disp, children[i], &wts);
		if (!(wts.map_state & IsViewable) || !onscreen(gtv, wts))
			continue;

		x = wts.x - gtv->vwin.x;
		y = wts.y - gtv->vwin.y;
		x2 = wts.width + 2 * wts.border_width;
		y2 = wts.height + 2 * wts.border_width;
		add_clip(gtv->cliprecs, &ncr, x, y, x2, y2);
	}
	XFree((char *) children);


	maketab(gtv, ncr);
	return ncr;
}

#endif


static void set_geo(Gtvscreen * gtv)
{
#ifndef NOCARD
	gint x, y, w, h;
	gint dx, dy;
	Boolean CHECK;
	gint oldx, oldy, oldw, oldh;
	GtkWidget *widget;
	GtkWidget *parent;

	g_return_if_fail(gtv != NULL);
	g_return_if_fail(IS_GTVSCREEN(gtv));

	oldx = gtv->vwin.x;
	oldy = gtv->vwin.y;
	oldw = gtv->vwin.width;
	oldh = gtv->vwin.height;

	CHECK = gtv->Change;

	gtv->vwin.flags = 0;
	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(gtv)))
		return;

	widget = GTK_WIDGET(gtv);
	parent = widget;
	gdk_window_get_deskrelative_origin(widget->window, &x, &y);

	w = widget->allocation.width;
	h = widget->allocation.height;

	if (oldh != h)
		CHECK = TRUE;
	if (oldw != w)
		CHECK = TRUE;

	dx = 0;
	dy = 0;
	if (w > gtv->vcap.maxwidth) {
		dx = (w - gtv->vcap.maxwidth) / 2;
		if (oldw != gtv->vcap.maxwidth) {
			CHECK = TRUE;
			gtv->vwin.width = gtv->vcap.maxwidth & ~3;
		}
	} else {
		gtv->vwin.width = w & ~3;
	}

	if (h > gtv->vcap.maxheight) {
		dy = (h - gtv->vcap.maxheight) / 2;
		if (oldh != gtv->vcap.maxheight) {
			CHECK = TRUE;
			gtv->vwin.height = gtv->vcap.maxheight & ~3;
		}
	} else
		gtv->vwin.height = h & ~3;

	if (oldx != x + dx) {
		gtv->Move = TRUE;
		CHECK = TRUE;
	}
	if (oldy != y + dy) {
		gtv->Move = TRUE;
		CHECK = TRUE;
	}

	if ((gtv->vbuf.depth == 24) && ((x + dx) & 3)) {
		parent->allocation.x =
		    parent->allocation.x - ((x + dx) & 3);
	} else if ((gtv->vbuf.depth == 16) && ((x + dx) & 1)) {
		parent->allocation.x =
		    parent->allocation.x - ((x + dx) & 1);
	} else if ((gtv->vbuf.depth == 15) && ((x + dx) & 1)) {
		parent->allocation.x =
		    parent->allocation.x - ((x + dx) & 1);
	}
	if ((y + dy) & 1) {
		parent->allocation.y =
		    parent->allocation.y - ((y + dy) & 1);
	}

	if (CHECK)
		get_clip(gtv);

	if (CHECK) {
		gtv->vwin.x = x + dx;
		gtv->vwin.y = y + dy;

		if (gtv->CAP_ON && gtv->vwin.width && gtv->vwin.height) {
			if (myioctl(gtv->fbttv, VIDIOCSWIN, &gtv->vwin))
				perror("set clipped window");
		}
	}

	gtv->Change = FALSE;
#endif
}


static GdkFilterReturn pConfigFilter(GdkXEvent * xevent,
				     GdkEvent * event, gpointer data)
{
	Window checkw;
	Gtvscreen *gtv;
	XEvent *xe;
	int type;
	XWindowAttributes wts1;

	xe = (XEvent *) xevent;
	if(IS_GTVSCREEN(data))
		gtv = GTVSCREEN(data);
	else 
		return GDK_FILTER_REMOVE;
		
	type = xe->type;
	checkw = xe->xmap.window;

#ifdef DEBUG
	printf("Filter:%d  Window:%d  %d  %d:\n", xe->type, 
	       checkw,gtv->lastrwin,GTK_WIDGET(gtv)->window);
#endif
	
	if (checkw == gtv->lastrwin || !gtv->OVL_ON)
		return GDK_FILTER_REMOVE;

	if (gtv->CAP_ON && 
	    type == ConfigureNotify &&
	    GTK_WIDGET_VISIBLE(GTK_WIDGET(gtv)) ) {
		XGetWindowAttributes(GDK_DISPLAY(), checkw, &wts1);
		if ( onscreen(gtv, wts1) ){
			set_capture(gtv, 0);
#ifndef NOCARD
			if (gtv->OVL_ON){
				get_clip(gtv);
				refresh(gtv);
			}
#endif
			set_capture(gtv, 1);
		}
	}
	return GDK_FILTER_REMOVE;
}


static gint pConfig(GtkWidget * widget, GdkEvent * event, gpointer data)
{
	Display *disp;
	static Gtvscreen *gtv = NULL;

	

	if (!gtv && data && (widget == GTK_WIDGET(data))
	    && (event == (GdkEvent *) NULL)) {
			gtv = (Gtvscreen *)data;
			return TRUE;
	}

	if (!event) return TRUE;
	if (gtv->OVL_ON && widget->window &&
	    GDK_WINDOW_XWINDOW(widget->window) == gtv->lastrwin)
		return FALSE;

#ifdef DEBUG
	printf("pConfig: %d\n", event->type);
#endif
	
	disp = (Display *) GDK_DISPLAY();


	if (gtv->CAP_ON && event->type == GDK_CONFIGURE) {
		set_capture(gtv, 0);
		set_geo(gtv);
		if (gtv->OVL_ON && event->type == GDK_CONFIGURE ){
			refresh(gtv);
		}
		set_capture(gtv, 1);
	}

	if (event->type == GDK_UNMAP || event->type == GDK_DESTROY ||
	    event->type == GDK_NO_EXPOSE) {
		if (gtv->CAP_ON) set_capture(gtv, 0);

	}

	return TRUE;
}

static void setmaxsize(Gtvscreen * gtv)
{
	g_return_if_fail(gtv != NULL);
	g_return_if_fail(IS_GTVSCREEN(gtv));
	if (gtv->vcap.maxwidth> MAXX) 
		gtv->vcap.maxwidth=MAXX; 
	if (gtv->vcap.maxheight> MAXY) 
		gtv->vcap.maxheight=MAXY; 
		
		
}


gboolean gtvscreen_set_values(Gtvscreen * gtv, gint val_type, gpointer value)
{
	gboolean pictmod;
	int old, new;

#ifdef DEBUG
	printf("set value: %i   %i\n", val_type, (int) value);
#endif

	pictmod = FALSE;
	switch (val_type) {

	case GTV_CAP:
		old = gtv->CAP_ON;
		gtv->CAP_ON = (int) value;
		if (old != gtv->CAP_ON) {
			switch (gtv->CAP_ON) {
			case 1:
				set_capture(gtv, 0);
				get_clip(gtv);
				gtv->Change = TRUE;
				set_geo(gtv);
				myioctl(gtv->fbttv, VIDIOCSCHAN, &gtv->vc);
				myioctl(gtv->fbttv, VIDIOCSPICT, &gtv->vp);
				set_capture(gtv, 1);
				break;
			case 0:
				set_capture(gtv, 0);
				refresh(gtv);

				break;
			}
			gtk_signal_emit(GTK_OBJECT(gtv),
					gtvscreen_signals[CHCAPTURE]);
		}
		break;

	case GTV_OVL:
		if (gtv->vcap.type & VID_TYPE_OVERLAY) {
			old = gtv->OVL_ON;
			gtv->OVL_ON = (int) value;
			if (old != gtv->OVL_ON && gtv->CAP_ON) {
				set_capture(gtv, 0);
				get_clip(gtv);
				gtv->Change = TRUE;
				set_geo(gtv);
				set_capture(gtv, 1);
				gtk_signal_emit(GTK_OBJECT(gtv),
						gtvscreen_signals
						[CHOVERLAY]);
			}
		}
		break;

	case GTV_DEVICE:
		if (!GTK_WIDGET_REALIZED(GTK_WIDGET(gtv)))
			strcpy(gtv->devname, (char *) value);
		break;

	case GTV_BPP:
		if (!GTK_WIDGET_REALIZED(GTK_WIDGET(gtv)))
			if (value)
				gtv->bpp = (int) value;
		break;

	case GTV_NORM:
		old = gtv->vc.norm;
		new = (int) value;
		if (new >= 0 && old != new) {
			gtv->vc.norm = new;
			if (gtv->CAP_ON) {
				gtv->Change = TRUE;
				set_capture(gtv, 0);
				setmaxsize(gtv);
				if (myioctl
				    (gtv->fbttv, VIDIOCSCHAN,
				     &gtv->vc)) perror("change norm");
				set_capture(gtv, 1);
				set_geo(gtv);
			}
			gtk_signal_emit(GTK_OBJECT(gtv),
					gtvscreen_signals[CHATTRIB]);
		}
		break;


	case GTV_INPUT:
		old = gtv->vc.channel;
		gtv->vc.channel = (int) value;
		if (old != gtv->vc.channel) {
			if (gtv->CAP_ON) {
				gtv->Change = TRUE;
				set_capture(gtv, 0);
				if (myioctl
				    (gtv->fbttv, VIDIOCSCHAN,
				     &gtv->vc)) perror("change input");
				set_geo(gtv);
				set_capture(gtv, 1);
			}
			gtk_signal_emit(GTK_OBJECT(gtv),
					gtvscreen_signals[CHINPUT]);
		}
		break;

	case GTV_BRIGHT:
		gtv->Change = TRUE;
		gtv->vp.brightness = (int) value;
		pictmod = TRUE;
		break;

	case GTV_COLOR:
		gtv->Change = TRUE;
		gtv->vp.colour = (int)value;
		pictmod = TRUE;
		break;

	case GTV_HUE:
		gtv->Change = TRUE;
		gtv->vp.hue = (int) value;
		pictmod = TRUE;
		break;

	case GTV_CONTRAST:
		gtv->Change = TRUE;
		gtv->vp.contrast = (int) value;
		pictmod = TRUE;
		break;

	case GTV_WHITE:
		gtv->Change = TRUE;
		gtv->vp.whiteness = (int)value;
		pictmod = TRUE;
		break;

	case GTV_VISNAME:	// maybe Ill fix that some day
		break;

	case GTV_REALW:
		if (!GTK_WIDGET_REALIZED(GTK_WIDGET(gtv)))
			gtv->realw = (int) value;
		return TRUE;
		break;

	case GTV_REMOTE:
		if (gtv->remote == -1) break;
		if (!gtv->remote && !(gtv->vcap.type & VID_TYPE_OVERLAY))
			gtv->remote = 1;
		else
			gtv->remote = (Boolean) ((int) value);
		break;

#ifdef HAVE_LIBXXF86VM
	case GTV_FULLS:
		set_capture(gtv, 0);
		gtv->next_vid = toggle_fullscreen(gtv->next_vid, gtv);
		set_capture(gtv, gtv->CAP_ON);
		break;
#endif
	}


	if (pictmod) {
		set_capture(gtv, 0);
		if (myioctl(gtv->fbttv, VIDIOCSPICT, &gtv->vp))
			perror("set picture params");
		set_capture(gtv, 1);
		set_geo(gtv);
		gtk_signal_emit(GTK_OBJECT(gtv),
				gtvscreen_signals[CHATTRIB]);
	}
	return TRUE;

}


gboolean gtvscreen_get_values(Gtvscreen * gtv, gint val_type, gint * value)
{

#ifdef DEBUG
	printf("get value: %i  %i\n", val_type, (int) value);
#endif

	switch (val_type) {
#ifndef NOCARD
		
	case GTV_COVL:
		*value = ((gtv->vcap.type & VID_TYPE_OVERLAY) ? 1:0); 
		break;

	case GTV_MAXX:
		setmaxsize(gtv);
		*value = gtv->vcap.maxwidth; 
		break;

	case GTV_MAXY:
		setmaxsize(gtv);
		*value = gtv->vcap.maxheight;
		break;

	case GTV_CAP:
		*value = gtv->CAP_ON;
		break;

	case GTV_OVL:
		*value = gtv->OVL_ON;
		break;

	case GTV_BPP:
		*value = gtv->bpp;
		break;

	case GTV_NORM:
		*value = gtv->vc.norm;
		break;

	case GTV_INPUT:
		*value = gtv->vc.channel;
		break;

	case GTV_BRIGHT:
		*value = gtv->vp.brightness;
		break;

	case GTV_COLOR:
		*value = gtv->vp.colour;
		break;

	case GTV_HUE:
		*value = gtv->vp.hue;
		break;

	case GTV_CONTRAST:
		*value = gtv->vp.contrast;
		break;

	case GTV_WHITE:
		*value = gtv->vp.whiteness;
		break;

	case GTV_VISNAME:
		value = (gint *) gtv->visname;
		break;

	case GTV_AUDIO:
		*value = gtv->va.audio;
		break;

	case GTV_REALW:
		*value = gtv->realw;
		break;

	case GTV_REMOTE:
		*value = gtv->remote;
		break;

	case GTV_NINPUT:
		setmaxsize(gtv);
		*value = gtv->vcap.channels;
		break;

	case GTV_NAUDIO:
		*value =
		    (gtv->va.mode & 1) + (gtv->va.mode & 2) / 2 +
		    (gtv->va.mode & 4) / 4 + (gtv->va.mode & 8) / 8;
		break;

#ifdef HAVE_LIBXXF86VM
	case GTV_FULLS:
		*value = gtv->isfull;
		break;
#endif

	}

#endif
	return TRUE;

}


static gint paint(gpointer data)
{
	gint x, y, w, h, d;
	GtkWidget *gwidget;
	Gtvscreen *gtv;

	gtv = GTVSCREEN(data);

	grab_image(gtv);
	gwidget = GTK_WIDGET(gtv);

	if (gtv->gc == NULL)
		return FALSE;

	gdk_window_get_geometry(gwidget->window, &x, &y, &w, &h, &d);
	gdk_window_get_deskrelative_origin(gwidget->window, &x, &y);
	if (w > MAXX)
		w = MAXX;

	if (h > MAXY)
		h = MAXY;

	gdk_draw_image(gwidget->window, gtv->gc, gtv->image, 0, 0,
		       gtv->vwin.x - x, gtv->vwin.y - y, w, h);

	gdk_flush();

	return TRUE;
}





static unsigned int which_format(int bpp)
{

	switch (bpp) {

	case 8:
		return VIDEO_PALETTE_HI240;	/* High 240 cube (BT848) */
		break;

	case 16:
		return VIDEO_PALETTE_RGB565;
		break;

	case 24:
		return VIDEO_PALETTE_RGB24;
		break;

	case 32:
		return VIDEO_PALETTE_RGB32;
		break;

	case 15:
		return VIDEO_PALETTE_RGB555;
		break;

	default:
		perror("Cant find color format");
		exit(1);
		break;
	}
}

static void grab_image(Gtvscreen * gtv)
{

	unsigned char *tm, *ctm, *imem;
	struct video_mmap gb;
	int i, fact, w;
	static int frame;

	g_return_if_fail(gtv != NULL);
	g_return_if_fail(IS_GTVSCREEN(gtv));

	imem = gtv->image->mem;
#ifndef NOCARD

	if (gtv->CAP_ON && !gtv->OVL_ON) {
		tm = gtv->TargetMem;

		gb.width = gtv->vwin.width;
		gb.height = gtv->vwin.height;

		gb.format = gtv->Format;

		if (gtv->start) {
			gtv->start = FALSE;
			frame = 0;
			gb.frame = 0;
			myioctl(gtv->fbttv, VIDIOCMCAPTURE, &gb);
			frame ^= 1;
		}

		gb.frame = frame;

		myioctl(gtv->fbttv, VIDIOCMCAPTURE, &gb);
		frame ^= 1;
		gb.frame = frame;
		ctm = tm + gtv->mbuf.offsets[gb.frame];
		myioctl(gtv->fbttv, VIDIOCSYNC, &gb.frame);

		fact = (int) (gtv->bpp / 8.0 + .5);
		w = gb.width * fact;
		for (i = 0; i < gb.height; i++)
			memcpy(imem + i * MAXX * fact, ctm + i * w, w);

#endif
	}
	if (gtv->stop) {
		frame ^= 1;
		gb.frame = frame;
		myioctl(gtv->fbttv, VIDIOCSYNC, &gb.frame);
	}
}





#ifdef HAVE_LIBXXF86VM
static int toggle_fullscreen(int nmode, Gtvscreen * gtv)
{
	int i, cmode, dummy, width, height, x, y, dx, dy;
	Display *disp;
	XF86VidModeModeLine modeline;
	GtkWidget *parent, *widget;
	GtkAllocation alloc;
	int found = FALSE;
	int fullx = 0, fully = 0;

	static XF86VidModeModeInfo **modelines;
	static int full;
	static int vmodes = 0;
	static int to, in, pb, aex;
	GdkCursor *ptr;
	GdkPixmap *pixi;
	char data[] = { 0, 0, 0, 0, 0, 0, 0, 0 };

	disp = (Display *) GDK_DISPLAY();
	widget = GTK_WIDGET(gtv);

	if (gtv->remote == 1)
		return -1;
	if (!vmodes) {
		XF86VidModeGetAllModeLines(disp, DefaultScreen(disp),
					   &vmodes, &modelines);


		for (i = 0; i < vmodes; i++)
		  if (FULLX1 == modelines[i]->hdisplay &&
		      FULLY1 == modelines[i]->vdisplay) {
		    found = TRUE;
		    fullx = FULLX1;
		    fully = FULLY1;
		    full = i;
		    break;
		  }

		if (!found)
		  for (i = 0; i < vmodes; i++)
		    if (FULLX2 == modelines[i]->hdisplay &&
			FULLY2 == modelines[i]->vdisplay) {
		      found = TRUE;
		      fullx = FULLX2;
		      fully = FULLY2;
		      full = i;
		      break;
		    }

		if (!found)
		  for (i = 0; i < vmodes; i++)
		    if (FULLX == modelines[i]->hdisplay &&
			FULLY == modelines[i]->vdisplay) {
		      found = TRUE;
		      fullx = FULLX;
		      fully = FULLY;
		      full = i;
		      break;
		    }

		XF86VidModeGetModeLine(disp, DefaultScreen(disp), &dummy,
				       &modeline);
		for (i = 0; i < vmodes; i++)
			if (modeline.hdisplay == modelines[i]->hdisplay
			    && modeline.vdisplay == modelines[i]->vdisplay)
				break;
		cmode = i;

		if (!found) {
			fprintf(stderr,
				_("Couldn't find fullscreen modeline %dx%d\n"),
				FULLX, FULLY);
			fprintf(stderr,
				_("Setting fullscreen to current mode\n"));
			full = cmode;
		}

		nmode = full;
		gtv->isfull = TRUE;
	} else {
		XF86VidModeGetModeLine(disp, DefaultScreen(disp), &i,
				       &modeline);
		for (i = 0; i < vmodes; i++)
			if (modeline.hdisplay == modelines[i]->hdisplay
			    && modeline.vdisplay == modelines[i]->vdisplay)
				break;
		cmode = i;
		if (gtv->isfull)
			gtv->isfull = FALSE;
		else
			gtv->isfull = TRUE;
	}
	if (nmode != cmode)
		XF86VidModeSwitchToMode(disp, DefaultScreen(disp),
					modelines[nmode]);

	if (gtv->isfull) {
		width = modelines[full]->htotal;
		height = modelines[full]->vtotal;

		parent = widget->parent;
		while (parent->parent) {
			parent = parent->parent;
		}

		alloc.width = FULLX;
		alloc.height = FULLY;
		alloc.x = 0;
		alloc.y = 0;
		gtk_widget_size_allocate(parent, &alloc);
		gdk_window_move(parent->window, 0, 0);

		gdk_window_get_deskrelative_origin(widget->window, &x, &y);
		dx = parent->allocation.width - gtv->vwin.width;
		dy = parent->allocation.height - gtv->vwin.height;
		gdk_window_move(parent->window, dx / 2- x - 1,
				dy / 2 - y - 5 - 1);
		alloc.width = fullx;
		alloc.height = fully;
		alloc.x = 0;
		alloc.y = 0;
		gtk_widget_size_allocate(widget, &alloc);


		XF86VidModeLockModeSwitch(disp, DefaultScreen(disp), TRUE);
		XWarpPointer(disp, None,
			     RootWindow(disp, DefaultScreen(disp)), 0, 0,
			     0, 0, 0, 0);

		XGetScreenSaver(disp, &to, &in, &pb, &aex);
		XSetScreenSaver(disp, 0, 0, DefaultBlanking,
				DefaultExposures);

		parent = widget->parent;
		while (parent->parent) {
			gtk_widget_hide(parent);
			parent = parent->parent;
		}
		pixi =
		    gdk_bitmap_create_from_data(widget->window, data, 8,
						8);
		ptr =
		    gdk_cursor_new_from_pixmap(pixi, pixi, gtv->colors,
					       gtv->colors, 0, 0);
		gdk_window_set_cursor(widget->window, ptr);
		gdk_cursor_destroy(ptr);
		gdk_pixmap_unref(pixi);


		return cmode;
	} else {
		parent = widget;
		while (parent->parent) {
			parent = parent->parent;
			gtk_widget_show(parent);
		}
		gdk_window_move(parent->window, 30, 30);
		ptr = gdk_cursor_new(GDK_RTL_LOGO);
		gdk_window_set_cursor(widget->window, ptr);
		gdk_cursor_destroy(ptr);

		gtk_widget_show(parent);
		XF86VidModeLockModeSwitch(disp, DefaultScreen(disp),
					  FALSE);
		XSetScreenSaver(disp, to, in, pb, aex);
		return full;
	}

}

#endif
