/*
 *
 * Copyright (C) 2000, 2001 Marcus Metzler 
 *            for convergence integrated media GmbH
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 * 

 * The author can be reached at marcus@convergence.de, 

 * the project's page is at http://linuxtv.org/dvb/
 */

#include <cdk/cdk.h>
#include "dvb_formats.h"
#include <sys/types.h>
#include <unistd.h>
#include <time.h>

unsigned int bitrates[3][16] =
{{0,32,64,96,128,160,192,224,256,288,320,352,384,416,448,0},
 {0,32,48,56,64,80,96,112,128,160,192,224,256,320,384,0},
 {0,32,40,48,56,64,80,96,112,128,160,192,224,256,320,0}};

uint32_t freq[4] = {441, 480, 320, 0};
char *frames[3] = {"I-Frame","P-Frame","B-Frame"};



uint32_t trans_pts_dts(uint8_t *pts)
{
        uint32_t wts;
        
        wts = (((pts[0] & 0x06) << 4) | 
               ((pts[1] & 0xFC) >> 2)) << 24; 
        wts |= (((pts[1] & 0x03) << 6) |
                ((pts[2] & 0xFC) >> 2)) << 16; 
        wts |= (((pts[2] & 0x02) << 6) |
                ((pts[3] & 0xFE) >> 1)) << 8;
        wts |= (((pts[3] & 0x01) << 7) |
                ((pts[4] & 0xFE) >> 1));
        return wts;
}

void init_ps(ps_packet *p)
{
        p->stuff_length=0xF8;
        p->data = NULL;
        p->sheader_length = 0;
        p->audio_bound = 0;
        p->video_bound = 0;
        p->npes = 0;
        p->mpeg = 2;
}

void kill_ps(ps_packet *p)
{
        if (p->data)
                free(p->data);
        init_ps(p);
}

void setlength_ps(ps_packet *p)
{
        short *ll;
        ll = (short *) p->sheader_llength;
        if (p->mpeg == 2)
                p->sheader_length = ntohs(*ll) - 6;
        else 
                p->sheader_length = ntohs(*ll);
}       

static void setl_ps(ps_packet *p)
{
        setlength_ps(p);
        p->data = (uint8_t *) malloc(p->sheader_length);
}

int cwrite_ps(uint8_t *buf, ps_packet *p, long length)
{
        long count,i;
        uint8_t headr1[4] = {0x00, 0x00, 0x01, 0xBA };
        uint8_t headr2[4] = {0x00, 0x00, 0x01, 0xBB };
        uint8_t buffy = 0xFF;

        
        memcpy(buf,headr1,4);
        count = 4;
        if (p->mpeg == 2){
                memcpy(buf+count,p->scr,6);
                count += 6;
                memcpy(buf+count,p->mux_rate,3);
                count += 3;
                memcpy(buf+count,&p->stuff_length,1);
                count++;
                for(i=0; i< (p->stuff_length & 3); i++){
                        memcpy(buf+count,&buffy,1);
                        count++;
                }
        } else {
                memcpy(buf+count,p->scr,5);
                count += 5;
                memcpy(buf+count,p->mux_rate,3);
                count += 3;
        }
        if (p->sheader_length){
                memcpy(buf+count,headr2,4);
                count += 4;
                memcpy(buf+count,p->sheader_llength,2);
                count += 2;
                if ( p->mpeg == 2){
                        memcpy(buf+count,p->rate_bound,3);
                        count += 3;
                        memcpy(buf+count,&p->audio_bound,1);
                        count++;
                        memcpy(buf+count,&p->video_bound,1);
                        count++;
                        memcpy(buf+count,&p->reserved,1);
                        count++;
                }
                memcpy(buf+count,p->data,p->sheader_length);
                count += p->sheader_length;
        }

        return count;
}

int write_ps_header(uint8_t *buf, 
                    uint32_t   SCR, 
                    long  muxr,
                    uint8_t    audio_bound,
                    uint8_t    fixed,
                    uint8_t    CSPS,
                    uint8_t    audio_lock,
                    uint8_t    video_lock,
                    uint8_t    video_bound,
                    uint8_t    stream1,
                    uint8_t    buffer1_scale,
                    uint32_t   buffer1_size,
                    uint8_t    stream2,
                    uint8_t    buffer2_scale,
                    uint32_t   buffer2_size)                    
{
        ps_packet p;
        uint8_t *pts;
        long lpts;
        init_ps(&p);
        
        lpts = htonl(SCR);
        pts = (uint8_t *) &lpts;

        
        p.mpeg = 2;
// SCR = 0
        p.scr[0] = 0x44;
        p.scr[1] = 0x00;
        p.scr[2] = 0x04;
        p.scr[3] = 0x00;
        p.scr[4] = 0x04;
        p.scr[5] = 0x01;
                
// SCR = PTS
        p.scr[0] = 0x44 | ((pts[0] >> 3)&0x18) | ((pts[0] >> 4)&0x03);
        p.scr[1] = 0x00 | ((pts[0] << 4)&0xF0) | ((pts[1] >> 4)&0x0F);
        p.scr[2] = 0x04 | ((pts[1] << 4)&0xF0) | ((pts[2] >> 4)&0x08)
                | ((pts[2] >> 5)&0x03);
        p.scr[3] = 0x00 | ((pts[2] << 3)&0xF8) | ((pts[3] >> 5)&0x07);
        p.scr[4] = 0x04 | ((pts[3] << 3)&0xF8);
        p.scr[5] = 0x01;
        
        p.mux_rate[0] = (uint8_t)(muxr >> 14);
        p.mux_rate[1] = (uint8_t)(0xff & (muxr >> 6));
        p.mux_rate[2] = (uint8_t)(0x03 | ((muxr & 0x3f) << 2));

        p.stuff_length = 0xF8;
        
        if (stream1 && stream2){
                p.sheader_llength[0] = 0x00;
                p.sheader_llength[1] = 0x0c;

                setl_ps(&p);
                
                p.rate_bound[0] = (uint8_t)(0x80 | (muxr >>15));
                p.rate_bound[1] = (uint8_t)(0xff & (muxr >> 7));
                p.rate_bound[2] = (uint8_t)(0x01 | ((muxr & 0x7f)<<1));

        
                p.audio_bound = (uint8_t)((audio_bound << 2)|(fixed << 1)|CSPS);
                p.video_bound = (uint8_t)((audio_lock << 7)|
                                     (video_lock << 6)|0x20|video_bound);
                p.reserved = (uint8_t)(0xFF >> 1);

                
                p.data[0] =  stream2;
                p.data[1] =  (uint8_t) (0xc0 | (buffer2_scale << 5) | 
                                   (buffer2_size >> 8));
                p.data[2] =  (uint8_t) (buffer2_size & 0xff);
                p.data[3] =  stream1;
                p.data[4] =  (uint8_t) (0xc0 | (buffer1_scale << 5) | 
                                   (buffer1_size >> 8));
                p.data[5] =  (uint8_t) (buffer1_size & 0xff);
                
                cwrite_ps(buf, &p, PS_HEADER_L2);
                kill_ps(&p);
                return PS_HEADER_L2;
        } else {
                cwrite_ps(buf, &p, PS_HEADER_L1);
                kill_ps(&p);
                return PS_HEADER_L1;
        }
}

ssize_t poll_write(int fd, uint8_t *buf, int count)
{
	struct pollfd pfd[2];
	int written,c;
	int skip=0;

	if (fd <0 ) return -1;
	pfd[0].fd = fd;
	pfd[0].events = POLLOUT;
	pfd[1].fd = STDIN_FILENO;
	pfd[1].events = POLLIN;

	written = 0;
	while(written < count && !skip){
		if (poll(pfd,2,1)){
			if (pfd[1].revents & POLLIN) skip = 1;
			
			if (pfd[0].revents & POLLOUT){
				c = write(fd,buf+written,count-written);
				if (c>0) written += c;
			}
		}
	}
	return written;
}

int pcm_play( int fd, uint8_t *samples, int sample_rate, uint8_t *pts, int len)
{
	static uint8_t header[9] = {0x00, 0x00, 0x01, 0xBD, 0x00, 0x00, 0x81,
				    0x00, 0x00};

	static uint8_t lpcm_header[7] = {0xA0, 0xff, 0x00, 0x00, 0x00, 
					 0x01, 0x80};

	uint16_t length;
	uint16_t olength;
	int diff = 0;
	
	//fprintf(stderr, "sample rate: %d\r", sample_rate);
	switch (sample_rate) {
	case 44100:
		lpcm_header[5] = 0x21;
		break;
	case 32000:
		lpcm_header[5] = 0x31;
		break;
	case 48000:
		lpcm_header[5] = 0x01;
		break;
	case 96000:
		lpcm_header[5] = 0x11;
		break;
		
	default:
		fprintf(stderr,"unsupported sample rate %d\n", sample_rate);
		exit(1);
	}

	while ( diff < len){
		length = 3+7;
		if (pts){
			header[7] = PTS_ONLY;
			header[8] = 0x05;
			length += 5;
		} else 	{
			header[7] = 0x00;
			header[8] = 0x00;
		}	
	
		olength = 2048 - length - 6;
		if (olength > len-diff) olength = len-diff;
		if (olength%2) olength--;
		length += olength;

		header[4] = ((uint8_t)(length >> 8) & 0xFF); 
		header[5] = ((uint8_t)(length) & 0xFF); 

		poll_write(fd, header, 9);
		if (pts) {
			poll_write(fd, pts, 5);
			pts = NULL;
		}
		poll_write(fd, lpcm_header, 7);
		poll_write(fd, samples+diff, olength);
		diff += olength;
	}

	return 0;
}


void get_pespts(uint8_t *av_pts,uint8_t *pts)
{
	
	pts[0] = 0x21 | 
		((av_pts[0] & 0xC0) >>5);
	pts[1] = ((av_pts[0] & 0x3F) << 2) |
		((av_pts[1] & 0xC0) >> 6);
	pts[2] = 0x01 | ((av_pts[1] & 0x3F) << 2) |
		((av_pts[2] & 0x80) >> 6);
	pts[3] = ((av_pts[2] & 0x7F) << 1) |
		((av_pts[3] & 0x80) >> 7);
	pts[4] = 0x01 | ((av_pts[3] & 0x7F) << 1);
}


uint16_t get_pid(uint8_t *pid)
{
	uint16_t pp;

	pp = (pid[0] & PID_MASK_HI) << 8;
	pp |= pid[1] &0xFF;
	return pp;
}




void reset_ipack(ipack *p)
{
	p->found = 0;
	p->cid = 0;
	p->plength = 0;
	p->flag1 = 0;
	p->flag2 = 0;
	p->hlength = 0;
	p->mpeg = 0;
	p->check = 0;
	p->which = 0;
	p->done = 0;
	p->count = 0;
	p->size = p->size_orig;
}

void init_ipack(ipack *p, int size,
		void (*func)(uint8_t *buf,  int size, void *priv), int ps)
{
	if ( !(p->buf = malloc(size)) ){
		printf("Couldn't allocate memroy for ipack\n");
		exit(1);
	}
	p->ps = ps;
	p->size_orig = size;
	p->func = func;
	reset_ipack(p);
	p->has_ai = 0;
	p->has_vi = 0;
	p->start = 0;
}


void free_ipack(ipack * p)
{
	if (p->buf) free(p->buf);
}

int get_vinfo(uint8_t *mbuf, int count, VideoInfo *vi, int pr)
{
	uint8_t *headr;
	int found = 0;
        int sw;
	int form = -1;
	int c = 0;

	while (found < 4 && c+4 < count){
		uint8_t *b;

		b = mbuf+c;
		if ( b[0] == 0x00 && b[1] == 0x00 && b[2] == 0x01
		     && b[3] == 0xb3) found = 4;
		else {
			c++;
		}
	}

	if (! found) return -1;
	c += 4;
	if (c+12 >= count) return -1;
	headr = mbuf+c;

	vi->horizontal_size	= ((headr[1] &0xF0) >> 4) | (headr[0] << 4);
	vi->vertical_size	= ((headr[1] &0x0F) << 8) | (headr[2]);
    
        sw = (int)((headr[3]&0xF0) >> 4) ;

        switch( sw ){
	case 1:
		if (pr)
			fprintf(stderr,"Videostream: ASPECT: 1:1");
		vi->aspect_ratio = 100;        
		break;
	case 2:
		if (pr)
			fprintf(stderr,"Videostream: ASPECT: 4:3");
                vi->aspect_ratio = 133;        
		break;
	case 3:
		if (pr)
			fprintf(stderr,"Videostream: ASPECT: 16:9");
                vi->aspect_ratio = 177;        
		break;
	case 4:
		if (pr)
			fprintf(stderr,"Videostream: ASPECT: 2.21:1");
                vi->aspect_ratio = 221;        
		break;

        case 5 ... 15:
		if (pr)
			fprintf(stderr,"Videostream: ASPECT: reserved");
                vi->aspect_ratio = 0;        
		break;

        default:
                vi->aspect_ratio = 0;        
                return -1;
	}

	if (pr)
		fprintf(stderr,"  Size = %dx%d",vi->horizontal_size,
			vi->vertical_size);

        sw = (int)(headr[3]&0x0F);

        switch ( sw ) {
	case 1:
		if (pr)
			fprintf(stderr,"  FRate: 23.976 fps");
                vi->framerate = 24000/1001.;
		form = -1;
		break;
	case 2:
		if (pr)
			fprintf(stderr,"  FRate: 24 fps");
                vi->framerate = 24;
		form = -1;
		break;
	case 3:
		if (pr)
			fprintf(stderr,"  FRate: 25 fps");
                vi->framerate = 25;
		form = VIDEO_MODE_PAL;
		break;
	case 4:
		if (pr)
			fprintf(stderr,"  FRate: 29.97 fps");
                vi->framerate = 30000/1001.;
		form = VIDEO_MODE_NTSC;
		break;
	case 5:
		if (pr)
			fprintf(stderr,"  FRate: 30 fps");
                vi->framerate = 30;
		form = VIDEO_MODE_NTSC;
		break;
	case 6:
		if (pr)
			fprintf(stderr,"  FRate: 50 fps");
                vi->framerate = 50;
		form = VIDEO_MODE_PAL;
		break;
	case 7:
		if (pr)
			fprintf(stderr,"  FRate: 60 fps");
                vi->framerate = 60;
		form = VIDEO_MODE_NTSC;
		break;
	}

	vi->bit_rate = 400*(((headr[4] << 10) & 0x0003FC00UL) 
			    | ((headr[5] << 2) & 0x000003FCUL) | 
			    (((headr[6] & 0xC0) >> 6) & 0x00000003UL));
	
	if (pr){
		fprintf(stderr,"  BRate: %.2f Mbit/s",(vi->bit_rate)/1000000.);
		fprintf(stderr,"\n");
	}
        vi->video_format = form;

	vi->off = c-4;
	return c-4;
}

extern unsigned int bitrates[3][16];
extern uint32_t freq[4];

int get_ainfo(uint8_t *mbuf, int count, AudioInfo *ai, int pr)
{
	uint8_t *headr;
	int found = 0;
	int c = 0;
	int fr =0;
	
	while (!found && c < count){
		uint8_t *b = mbuf+c;

		if ( b[0] == 0xff && (b[1] & 0xf8) == 0xf8)
			found = 1;
		else {
			c++;
		}
	}	

	if (!found) return -1;

	if (c+3 >= count) return -1;
        headr = mbuf+c;

	ai->layer = (headr[1] & 0x06) >> 1;

        if (pr)
		fprintf(stderr,"Audiostream: Layer: %d", 4-ai->layer);


	ai->bit_rate = bitrates[(3-ai->layer)][(headr[2] >> 4 )]*1000;

	if (pr){
		if (ai->bit_rate == 0)
			fprintf (stderr,"  Bit rate: free");
		else if (ai->bit_rate == 0xf)
			fprintf (stderr,"  BRate: reserved");
		else
			fprintf (stderr,"  BRate: %d kb/s", ai->bit_rate/1000);
	}

	fr = (headr[2] & 0x0c ) >> 2;
	ai->frequency = freq[fr]*100;
	
	if (pr){
		if (ai->frequency == 3)
			fprintf (stderr, "  Freq: reserved\n");
		else
			fprintf (stderr,"  Freq: %2.1f kHz\n", 
				 ai->frequency/1000.);
	}
	ai->off = c;
	return c;
}

unsigned int ac3_bitrates[32] =
    {32,40,48,56,64,80,96,112,128,160,192,224,256,320,384,448,512,576,640,
     0,0,0,0,0,0,0,0,0,0,0,0,0};

uint32_t ac3_freq[4] = {480, 441, 320, 0};
uint32_t ac3_frames[3][32] =
    {{64,80,96,112,128,160,192,224,256,320,384,448,512,640,768,896,1024,
      1152,1280,0,0,0,0,0,0,0,0,0,0,0,0,0},
     {69,87,104,121,139,174,208,243,278,348,417,487,557,696,835,975,1114,
      1253,1393,0,0,0,0,0,0,0,0,0,0,0,0,0},
     {96,120,144,168,192,240,288,336,384,480,576,672,768,960,1152,1344,
      1536,1728,1920,0,0,0,0,0,0,0,0,0,0,0,0,0}}; 

int get_ac3info(uint8_t *mbuf, int count, AudioInfo *ai, int pr)
{
	uint8_t *headr;
	int found = 0;
	int c = 0;
	uint8_t frame;
	int fr = 0;

	while ( !found  && c < count){
		uint8_t *b = mbuf+c;
		if ( b[0] == 0x0b &&  b[1] == 0x77 )
			found = 1;
		else {
			c++;
		}
	}	


	if (!found){
		return -1;
	}
	ai->off = c;

	if (c+5 >= count) return -1;

	ai->layer = 0;  // 0 for AC3
        headr = mbuf+c+2;

	frame = (headr[2]&0x3f);
	ai->bit_rate = ac3_bitrates[frame>>1]*1000;

	if (pr) fprintf (stderr,"  BRate: %d kb/s", ai->bit_rate/1000);

	fr = (headr[2] & 0xc0 ) >> 6;
	ai->frequency = freq[fr]*100;
	if (pr) fprintf (stderr,"  Freq: %d Hz\n", ai->frequency);

	ai->framesize = ac3_frames[fr][frame >> 1];
	if ((frame & 1) &&  (fr == 1)) ai->framesize++;
	ai->framesize = ai->framesize << 1;
	if (pr) fprintf (stderr,"  Framesize %d\n", ai->framesize);

	return c;
}


void ps_pes(ipack *p)
{
	int check;
	uint8_t pbuf[PS_HEADER_L2];
	static int muxr = 0;
	static int ai = 0;
	static int vi = 0;
	static int start = 0;
	static uint32_t SCR = 0;

	if (p->mpeg == 2){
		switch(p->buf[3]){
		case VIDEO_STREAM_S ... VIDEO_STREAM_E:
			if (!p->has_vi){
				if(get_vinfo(p->buf, p->count, &p->vi,0) >=0) {
					p->has_vi = 1;
					vi = p->vi.bit_rate;
				}
			} 			
			break;

		case AUDIO_STREAM_S ... AUDIO_STREAM_E:
			if (!p->has_ai){
				if(get_ainfo(p->buf, p->count, &p->ai,0) >=0) {
					p->has_ai = 1;
					ai = p->ai.bit_rate;
				}
			} 
			break;
		}

		if (p->has_vi && vi && !muxr){
			muxr = (vi+ai)/400;
		}

		if ( start && muxr && (p->buf[7] & PTS_ONLY) && (p->has_ai || 
				       p->buf[9+p->buf[8]+4] == 0xb3)){  
			SCR = trans_pts_dts(p->pts)-3600;
			
			check = write_ps_header(pbuf,
						SCR,
						muxr, 1, 0, 0, 1, 1, 1, 
						0, 0, 0, 0, 0, 0);

			p->func(pbuf, check , p->data);
		}

		if (muxr && !start && vi){
			SCR = trans_pts_dts(p->pts)-3600;
			check = write_ps_header(pbuf,
						SCR, 
						muxr, 1, 0, 0, 1, 1, 1, 
						0xC0, 0, 64, 0xE0, 1, 460);
			start = 1;
			p->func(pbuf, check , p->data);
		}

		if (start)
			p->func(p->buf, p->count, p->data);
	}
}

void send_ipack(ipack *p)
{
	int streamid=0;
	int off;
	int ac3_off = 0;
	AudioInfo ai;
	int nframes= 0;
	int f=0;

	if (p->count < 10) return;
	p->buf[3] = p->cid;
	p->buf[4] = (uint8_t)(((p->count-6) & 0xFF00) >> 8);
	p->buf[5] = (uint8_t)((p->count-6) & 0x00FF);

	if (p->cid == PRIVATE_STREAM1){

		off = 9+p->buf[8];
		streamid = p->buf[off];
		if ((streamid & 0xF8) == 0x80){
			ai.off = 0;
			ac3_off = ((p->buf[off+2] << 8)| p->buf[off+3]);
			if (ac3_off < p->count)
				f=get_ac3info(p->buf+off+3+ac3_off, 
					      p->count-ac3_off, &ai,0);
			if ( !f ){
				nframes = (p->count-off-3-ac3_off)/ 
					ai.framesize + 1;
				p->buf[off+2] = (ac3_off >> 8)& 0xFF;
				p->buf[off+3] = (ac3_off)& 0xFF;
				p->buf[off+1] = nframes;
				
				ac3_off +=  nframes * ai.framesize - p->count;
			}
		}
	} 
	
	if (p->ps) ps_pes(p);
	else p->func(p->buf, p->count, p->data);
	
	switch ( p->mpeg ){
	case 2:		
		p->buf[6] = 0x80;
		p->buf[7] = 0x00;
		p->buf[8] = 0x00;
		p->count = 9;

		if (p->cid == PRIVATE_STREAM1 && (streamid & 0xF8)==0x80 ){
			p->count += 4;
			p->buf[9] = streamid;
			p->buf[10] = (ac3_off >> 8)& 0xFF;
			p->buf[11] = (ac3_off)& 0xFF;
			p->buf[12] = 0;
		}

		break;
	case 1:
		p->buf[6] = 0x0F;
		p->count = 7;
		break;
	}
}


void write_ipack(ipack *p, uint8_t *data, int count)
{
	uint8_t headr[3] = { 0x00, 0x00, 0x01} ;
	int diff =0;

	if (p->count < 6){
		if (trans_pts_dts(p->pts) > trans_pts_dts(p->last_pts))
			memcpy(p->last_pts, p->pts, 5);
		p->count = 0;
		memcpy(p->buf+p->count, headr, 3);
		p->count += 6;
	}
	if ( p->size == p->size_orig && p->plength &&
	     (diff = 6+p->plength - p->found + p->count +count) > p->size &&
	     diff < 3*p->size/2){
		
			p->size = diff/2;
//			fprintf(stderr,"size: %d \n",p->size);
	}
	if (p->count + count < p->size){
		memcpy(p->buf+p->count, data, count);
		p->count += count;
	} else {
		int rest = p->size - p->count;
		if (rest < 0) rest = 0;
		memcpy(p->buf+p->count, data, rest);
		p->count += rest;
		send_ipack(p);
		if (count - rest > 0)
			write_ipack(p, data+rest, count-rest);
 	}
}

void instant_repack (uint8_t *buf, int count, ipack *p)
{

	int l;
	unsigned short *pl;
	int c=0;

	while (c < count && (p->mpeg == 0 ||
			     (p->mpeg == 1 && p->found < 7) ||
			     (p->mpeg == 2 && p->found < 9))
	       &&  (p->found < 5 || !p->done)){
		switch ( p->found ){
		case 0:
		case 1:
			if (buf[c] == 0x00) p->found++;
			else p->found = 0;
			c++;
			break;
		case 2:
			if (buf[c] == 0x01) p->found++;
			else if (buf[c] == 0){
				p->found = 2;
			} else p->found = 0;
			c++;
			break;
		case 3:
			p->cid = 0;
			switch (buf[c]){
			case PROG_STREAM_MAP:
			case PRIVATE_STREAM2:
			case PROG_STREAM_DIR:
			case ECM_STREAM     :
			case EMM_STREAM     :
			case PADDING_STREAM :
			case DSM_CC_STREAM  :
			case ISO13522_STREAM:
				p->done = 1;
			case PRIVATE_STREAM1:
			case VIDEO_STREAM_S ... VIDEO_STREAM_E:
			case AUDIO_STREAM_S ... AUDIO_STREAM_E:
				p->found++;
				p->cid = buf[c];
				c++;
				break;
			default:
				p->found = 0;
				break;
			}
			break;
			

		case 4:
			if (count-c > 1){
				pl = (unsigned short *) (buf+c);
				p->plength =  ntohs(*pl);
				p->plen[0] = buf[c];
				c++;
				p->plen[1] = buf[c];
				c++;
				p->found+=2;
			} else {
				p->plen[0] = buf[c];
				p->found++;
				return;
			}
			break;
		case 5:
			p->plen[1] = buf[c];
			c++;
			pl = (unsigned short *) p->plen;
			p->plength = ntohs(*pl);
			p->found++;
			break;


		case 6:
			if (!p->done){
				p->flag1 = buf[c];
				c++;
				p->found++;
				if ( (p->flag1 & 0xC0) == 0x80 ) p->mpeg = 2;
				else {
					p->hlength = 0;
					p->which = 0;
					p->mpeg = 1;
					p->flag2 = 0;
				}
			}
			break;

		case 7:
			if ( !p->done && p->mpeg == 2){
				p->flag2 = buf[c];
				c++;
				p->found++;
			}	
			break;

		case 8:
			if ( !p->done && p->mpeg == 2){
				p->hlength = buf[c];
				c++;
				p->found++;
			}
			break;
			
		default:

			break;
		}
	}

	if (c == count) return;

	if (!p->plength) p->plength = MMAX_PLENGTH-6;


	if ( p->done || ((p->mpeg == 2 && p->found >= 9)  || 
	     (p->mpeg == 1 && p->found >= 7)) ){
		switch (p->cid){
			
		case AUDIO_STREAM_S ... AUDIO_STREAM_E:			
		case VIDEO_STREAM_S ... VIDEO_STREAM_E:
		case PRIVATE_STREAM1:
			
			if (p->mpeg == 2 && p->found == 9){
				write_ipack(p, &p->flag1, 1);
				write_ipack(p, &p->flag2, 1);
				write_ipack(p, &p->hlength, 1);
			}

			if (p->mpeg == 1 && p->found == 7){
				write_ipack(p, &p->flag1, 1);
			}


			if (p->mpeg == 2 && (p->flag2 & PTS_ONLY) &&  
			    p->found < 14){
				while (c < count && p->found < 14){
					p->pts[p->found-9] = buf[c];
					write_ipack(p, buf+c, 1);
					c++;
					p->found++;
				}
				if (c == count) return;
			}

			if (p->mpeg == 1 && p->which < 2000){

				if (p->found == 7) {
					p->check = p->flag1;
					p->hlength = 1;
				}

				while (!p->which && c < count && 
				       p->check == 0xFF){
					p->check = buf[c];
					write_ipack(p, buf+c, 1);
					c++;
					p->found++;
					p->hlength++;
				}

				if ( c == count) return;
				
				if ( (p->check & 0xC0) == 0x40 && !p->which){
					p->check = buf[c];
					write_ipack(p, buf+c, 1);
					c++;
					p->found++;
					p->hlength++;

					p->which = 1;
					if ( c == count) return;
					p->check = buf[c];
					write_ipack(p, buf+c, 1);
					c++;
					p->found++;
					p->hlength++;
					p->which = 2;
					if ( c == count) return;
				}

				if (p->which == 1){
					p->check = buf[c];
					write_ipack(p, buf+c, 1);
					c++;
					p->found++;
					p->hlength++;
					p->which = 2;
					if ( c == count) return;
				}
				
				if ( (p->check & 0x30) && p->check != 0xFF){
					p->flag2 = (p->check & 0xF0) << 2;
					p->pts[0] = p->check;
					p->which = 3;
				} 

				if ( c == count) return;
				if (p->which > 2){
					if ((p->flag2 & PTS_DTS_FLAGS)
					    == PTS_ONLY){
						while (c < count && 
						       p->which < 7){
							p->pts[p->which-2] =
								buf[c];
							write_ipack(p,buf+c,1);
							c++;
							p->found++;
							p->which++;
							p->hlength++;
						}
						if ( c == count) return;
					} else if ((p->flag2 & PTS_DTS_FLAGS) 
						   == PTS_DTS){
						while (c < count && 
						       p->which< 12){
							if (p->which< 7)
								p->pts[p->which
								      -2] =
									buf[c];
							write_ipack(p,buf+c,1);
							c++;
							p->found++;
							p->which++;
							p->hlength++;
						}
						if ( c == count) return;
					}
					p->which = 2000;
				}
							
			}

			while (c < count && p->found < p->plength+6){
				l = count -c;
				if (l+p->found > p->plength+6)
					l = p->plength+6-p->found;
				write_ipack(p, buf+c, l);
				p->found += l;
				c += l;
			}	
		
			break;
		}


		if ( p->done ){
			if( p->found + count - c < p->plength+6){
				p->found += count-c;
				c = count;
			} else {
				c += p->plength+6 - p->found;
				p->found = p->plength+6;
			}
		}

		if (p->plength && p->found == p->plength+6) {
			send_ipack(p);
			reset_ipack(p);
			if (c < count)
				instant_repack(buf+c, count-c, p);
		}
	}
	return;
}


int write_ts_header(uint16_t pid, uint8_t *counter, int pes_start, 
		    uint8_t *buf, uint8_t length)
{
	int i;
	int c = 0;
	int fill;
	uint8_t tshead[4] = { 0x47, 0x00, 0x00, 0x10}; 
        

	fill = TS_SIZE-4-length;
        if (pes_start) tshead[1] = 0x40;
	if (fill) tshead[3] = 0x30;
        tshead[1] |= (uint8_t)((pid & 0x1F00) >> 8);
        tshead[2] |= (uint8_t)(pid & 0x00FF);
        tshead[3] |= ((*counter)++ & 0x0F) ;
        memcpy(buf,tshead,4);
	c+=4;


	if (fill){
		buf[4] = fill-1;
		c++;
		if (fill >1){
			buf[5] = 0x00;
			c++;
		}
		for ( i = 6; i < fill+4; i++){
			buf[i] = 0xFF;
			c++;
		}
	}

        return c;
}


int write_pes_header(uint8_t id,int length , long PTS, uint8_t *obuf, 
		     int stuffing)
{
	uint8_t le[2];
	uint8_t dummy[3];
	uint8_t *pts;
	uint8_t ppts[5];
	long lpts;
	int c;
	uint8_t headr[3] = {0x00, 0x00, 0x01};
	
	lpts = htonl(PTS);
	pts = (uint8_t *) &lpts;
	
	get_pespts(pts,ppts);

	c = 0;
	memcpy(obuf+c,headr,3);
	c += 3;
	memcpy(obuf+c,&id,1);
	c++;

	le[0] = 0;
	le[1] = 0;
	length -= 6+stuffing;

	le[0] |= ((uint8_t)(length >> 8) & 0xFF); 
	le[1] |= ((uint8_t)(length) & 0xFF); 
	memcpy(obuf+c,le,2);
	c += 2;

	if (id == PADDING_STREAM){
		memset(obuf+c,0xff,length);
		c+= length;
		return c;
	}

	dummy[0] = 0x80;
	dummy[1] = 0;
	dummy[2] = 0;
	if (PTS){
		dummy[1] |= PTS_ONLY;
		dummy[2] = 5+stuffing;
	}
	memcpy(obuf+c,dummy,3);
	c += 3;
	memset(obuf+c,0xFF,stuffing);

	if (PTS){
		memcpy(obuf+c,ppts,5);
		c += 5;
	}
	
	return c;
}


void init_p2p(p2p *p, void (*func)(uint8_t *buf, int count, void *data), 
	      int repack){
	p->found = 0;
	p->cid = 0;
	p->mpeg = 0;
	p->done = 0;
	p->func = func;
	p->bigend_repack = 0;
	p->repack = 0; 
	if ( repack < MAX_PLENGTH && repack > 265 ){
		p->repack = repack-6;
		p->bigend_repack = (uint16_t)htons((short)((repack-6) & 0xFFFF));
	} 
	
	if( !(p->buf = (char *) malloc(sizeof(char)*MMAX_PLENGTH))){
		fprintf(stderr,"Not enough memory for ts_transform\n");
	}
	if( repack && !(p->buf2 = (char *) malloc(sizeof(char)*p->repack))){
		fprintf(stderr,"Not enough memory for ts_transform\n");
	}

	
	memset(p->buf,0,MMAX_PLENGTH);

}

void free_p2p(p2p *p)
{
	free(p->buf);
	free(p->buf2);
	p->buf = NULL;
	p->buf2 = NULL;
}


void pes_repack(p2p *p)
{
	int count = 0;
	int repack = p->repack;
	int rest = p->plength;
	int bfill = 0;
	int diff;
	uint16_t length;
	uint8_t *buf = p->buf2;
	

	if (rest <= 0) {
		//fprintf(stderr,"Error in repack\n");
		return;
	}
	if (!repack){
		fprintf(stderr,"forgot to set repack size\n");
		return;
	}
	if (p->plength == repack){
		memcpy(p->buf+4,(char *)&p->bigend_repack,2);
		p->func(p->buf, repack+6, p);
		return;
	}
	
	buf[0] = 0x00;
	buf[1] = 0x00;
	buf[2] = 0x01;
	buf[3] = p->cid;
	memcpy(buf+4,(char *)&p->bigend_repack,2);
	memset(buf+6,0, repack-6);

	if (p->mpeg == 2){

		if ( rest > repack){
			memcpy(p->buf+4,(char *)&p->bigend_repack,2);
			p->func(p->buf, repack+6, p->data);
			count += repack+6;
			rest -= repack;
		} else {
			memcpy(buf,p->buf,9+p->hlength);
			bfill = p->hlength;
			count += 9+p->hlength;
			rest -= p->hlength+3;
		}

		while (rest >= repack-3){
			memset(buf+6,0,repack-6);
			buf[6] = 0x80;
			buf[7] = 0x00;
			buf[8] = 0x00;
			memcpy(buf+9,p->buf+count,repack-3);
			rest -= repack-3;
			count += repack-3;
			p->func(buf, repack+6, p->data);
		}
		
		if (rest){
			diff = (repack - 3) - rest - bfill;
			if (!bfill){
				buf[6] = 0x80;
				buf[7] = 0x00;
				buf[8] = 0x00;
			}

			if ( diff < PES_MIN){
				length = repack; 
				buf[4] = (uint8_t)((length & 0xFF00) >> 8);
				buf[5] = (uint8_t)(length & 0x00FF);
				buf[8] = (uint8_t)(bfill+diff);
				memset(buf+9+bfill,0xFF,diff);
				memcpy(buf+9+bfill+diff,p->buf+count,rest);
			} else {
				length = rest+ bfill+3; 
				buf[4] = (uint8_t)((length & 0xFF00) >> 8);
				buf[5] = (uint8_t)(length & 0x00FF);
				memcpy(buf+9+bfill,p->buf+count,rest);
				bfill += rest+9;
				write_pes_header( PADDING_STREAM, diff, 0,
						  buf+bfill, 0);
			}
			p->func(buf, repack+6, p->data);
		}
	}	

	if (p->mpeg == 1){

		if ( rest > repack){
			memcpy(p->buf+4,(char *)&p->bigend_repack,2);
			p->func(p->buf, repack+6, p->data);
			count += repack+6;
			rest -= repack;
		} else {
			memcpy(buf,p->buf,6+p->hlength);
			bfill = p->hlength;
			count += 6;
			rest -= p->hlength;
		}

		while (rest >= repack-1){
			memset(buf+6,0,MAX_PLENGTH-6);
			buf[6] = 0x0F;
			memcpy(buf+7,p->buf+count,repack-1);
			rest -= repack-1;
			count += repack-1;
			p->func(buf, repack+6, p->data);
		}
		
		if (rest){
			diff = repack - 1 - rest - bfill;

			if ( diff < PES_MIN){
				length = repack; 
				buf[4] = (uint8_t)((length & 0xFF00) >> 8);
				buf[5] = (uint8_t)(length & 0x00FF);
				memset(buf+6,0xFF,diff);
				if (!bfill){
					buf[6+diff] = 0x0F;
				}
				memcpy(buf+7+diff,p->buf+count,rest+bfill);
			} else {
				length = rest+ bfill+1; 
				buf[4] = (uint8_t)((length & 0xFF00) >> 8);
				buf[5] = (uint8_t)(length & 0x00FF);
				if (!bfill){
					buf[6] = 0x0F;
					memcpy(buf+7,p->buf+count,rest);
					bfill = rest+7;
				} else {
					memcpy(buf+6,p->buf+count,rest+bfill);
					bfill += rest+6;
				}
				write_pes_header( PADDING_STREAM, diff, 0,
						  buf+bfill, 0);
			}
			p->func(buf, repack+6, p->data);
		}
	}	
}


void get_pes (uint8_t *buf, int count, p2p *p, void (*func)(p2p *p))
{

	int l;
	unsigned short *pl;
	int c=0;

	uint8_t headr[3] = { 0x00, 0x00, 0x01} ;


	while (c < count && (p->mpeg == 0 ||
			     (p->mpeg == 1 && p->found < 7) ||
			     (p->mpeg == 2 && p->found < 9))
	       &&  (p->found < 5 || !p->done)){
		switch ( p->found ){
		case 0:
		case 1:
			if (buf[c] == 0x00) p->found++;
			else p->found = 0;
			c++;
			break;
		case 2:
			if (buf[c] == 0x01) p->found++;
			else if (buf[c] == 0){
				p->found = 2;
			} else p->found = 0;
			c++;
			break;
		case 3:
			p->cid = 0;
			switch (buf[c]){
			case PROG_STREAM_MAP:
			case PRIVATE_STREAM2:
			case PROG_STREAM_DIR:
			case ECM_STREAM     :
			case EMM_STREAM     :
			case PADDING_STREAM :
			case DSM_CC_STREAM  :
			case ISO13522_STREAM:
				p->done = 1;
			case PRIVATE_STREAM1:
			case VIDEO_STREAM_S ... VIDEO_STREAM_E:
			case AUDIO_STREAM_S ... AUDIO_STREAM_E:
				p->found++;
				p->cid = buf[c];
				c++;
				break;
			default:
				p->found = 0;
				break;
			}
			break;
			

		case 4:
			if (count-c > 1){
				pl = (unsigned short *) (buf+c);
				p->plength =  ntohs(*pl);
				p->plen[0] = buf[c];
				c++;
				p->plen[1] = buf[c];
				c++;
				p->found+=2;
			} else {
				p->plen[0] = buf[c];
				p->found++;
				return;
			}
			break;
		case 5:
			p->plen[1] = buf[c];
			c++;
			pl = (unsigned short *) p->plen;
			p->plength = ntohs(*pl);
			p->found++;
			break;


		case 6:
			if (!p->done){
				p->flag1 = buf[c];
				c++;
				p->found++;
				if ( (p->flag1 & 0xC0) == 0x80 ) p->mpeg = 2;
				else {
					p->hlength = 0;
					p->which = 0;
					p->mpeg = 1;
					p->flag2 = 0;
				}
			}
			break;

		case 7:
			if ( !p->done && p->mpeg == 2){
				p->flag2 = buf[c];
				c++;
				p->found++;
			}	
			break;

		case 8:
			if ( !p->done && p->mpeg == 2){
				p->hlength = buf[c];
				c++;
				p->found++;
			}
			break;
			
		default:

			break;
		}
	}

	if (!p->plength) p->plength = MMAX_PLENGTH-6;


	if ( p->done || ((p->mpeg == 2 && p->found >= 9)  || 
	     (p->mpeg == 1 && p->found >= 7)) ){
		switch (p->cid){
			
		case AUDIO_STREAM_S ... AUDIO_STREAM_E:			
		case VIDEO_STREAM_S ... VIDEO_STREAM_E:
		case PRIVATE_STREAM1:

			memcpy(p->buf, headr, 3);
			p->buf[3] = p->cid;
			memcpy(p->buf+4,p->plen,2);

			if (p->mpeg == 2 && p->found == 9){
				p->buf[6] = p->flag1;
				p->buf[7] = p->flag2;
				p->buf[8] = p->hlength;
			}

			if (p->mpeg == 1 && p->found == 7){
				p->buf[6] = p->flag1;
			}


			if (p->mpeg == 2 && (p->flag2 & PTS_ONLY) &&  
			    p->found < 14){
				while (c < count && p->found < 14){
					p->pts[p->found-9] = buf[c];
					p->buf[p->found] = buf[c];
					c++;
					p->found++;
				}
				if (c == count) return;
			}

			if (p->mpeg == 1 && p->which < 2000){

				if (p->found == 7) {
					p->check = p->flag1;
					p->hlength = 1;
				}

				while (!p->which && c < count && 
				       p->check == 0xFF){
					p->check = buf[c];
					p->buf[p->found] = buf[c];
					c++;
					p->found++;
					p->hlength++;
				}

				if ( c == count) return;
				
				if ( (p->check & 0xC0) == 0x40 && !p->which){
					p->check = buf[c];
					p->buf[p->found] = buf[c];
					c++;
					p->found++;
					p->hlength++;

					p->which = 1;
					if ( c == count) return;
					p->check = buf[c];
					p->buf[p->found] = buf[c];
					c++;
					p->found++;
					p->hlength++;
					p->which = 2;
					if ( c == count) return;
				}

				if (p->which == 1){
					p->check = buf[c];
					p->buf[p->found] = buf[c];
					c++;
					p->found++;
					p->hlength++;
					p->which = 2;
					if ( c == count) return;
				}
				
				if ( (p->check & 0x30) && p->check != 0xFF){
					p->flag2 = (p->check & 0xF0) << 2;
					p->pts[0] = p->check;
					p->which = 3;
				} 

				if ( c == count) return;
				if (p->which > 2){
					if ((p->flag2 & PTS_DTS_FLAGS)
					    == PTS_ONLY){
						while (c < count && 
						       p->which < 7){
							p->pts[p->which-2] =
								buf[c];
							p->buf[p->found] = 
								buf[c];
							c++;
							p->found++;
							p->which++;
							p->hlength++;
						}
						if ( c == count) return;
					} else if ((p->flag2 & PTS_DTS_FLAGS) 
						   == PTS_DTS){
						while (c < count && 
						       p->which< 12){
							if (p->which< 7)
								p->pts[p->which
								      -2] =
									buf[c];
							p->buf[p->found] = 
								buf[c];
							c++;
							p->found++;
							p->which++;
							p->hlength++;
						}
						if ( c == count) return;
					}
					p->which = 2000;
				}
							
			}

			while (c < count && p->found < p->plength+6){
				l = count -c;
				if (l+p->found > p->plength+6)
					l = p->plength+6-p->found;
				memcpy(p->buf+p->found, buf+c, l);
				p->found += l;
				c += l;
			}			
			if(p->found == p->plength+6)
				func(p);
			
			break;
		}


		if ( p->done ){
			if( p->found + count - c < p->plength+6){
				p->found += count-c;
				c = count;
			} else {
				c += p->plength+6 - p->found;
				p->found = p->plength+6;
			}
		}

		if (p->plength && p->found == p->plength+6) {
			p->found = 0;
			p->done = 0;
			p->plength = 0;
			memset(p->buf, 0, MAX_PLENGTH);
			if (c < count)
				get_pes(buf+c, count-c, p, func);
		}
	}
	return;
}


void pes_in_ts(p2p *p)
{
	int l, pes_start;
	uint8_t obuf[TS_SIZE];
	long int c = 0;
	int length = p->plength+6;
	uint16_t pid;
	uint8_t *counter;
	uint8_t tspid0[TS_SIZE] = { 
                0x47, 0x40, 0x00, 0x10, 0x00, 0x00, 0xb0, 0x11, 
                0x00, 0x00, 0xcb, 0x00, 0x00, 0x00, 0x00, 0xe0, 
                0x10, 0x00, 0x01, 0xe4, 0x00, 0x2a, 0xd6, 0x1a, 
                0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff
        };

	uint8_t tspid1[TS_SIZE] = { 
                0x47, 0x44, 0x00, 0x10, 0x00, 0x02, 0xb0, 0x1c,
                0x00, 0x01, 0xcb, 0x00, 0x00, 0xe0, 0xa0, 0xf0, 
                0x05, 0x48, 0x03, 0x01, 0x00, 0x00, 0x02, 0xe0,
                0xa0, 0xf0, 0x00, 0x03, 0xe0, 0x50, 0xf0, 0x00, 
                0xae, 0xea, 0x4e, 0x48, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff
        };

	pes_start = 1;
	switch ( p->cid ) {
	case AUDIO_STREAM_S ... AUDIO_STREAM_E:			
		pid = p->pida;
		counter = &p->acounter;
		break;
	case VIDEO_STREAM_S ... VIDEO_STREAM_E:
		pid = p->pidv;
		counter = &p->acounter;

		tspid0[3] |= (p->count0++) 
			& 0x0F ;
		tspid1[3] |= (p->count1++) 
			& 0x0F ;
	
		tspid1[24]  = p->pidv;
		tspid1[23] |= (p->pidv >> 8) & 0x3F;
		tspid1[29]  = p->pida;
		tspid1[28] |= (p->pida >> 8) & 0x3F;
		
		p->func(tspid0,188,p->data);
		p->func(tspid1,188,p->data);
		break;
	default:
		return;
	}

	while ( c < length ){
		memset(obuf,0,TS_SIZE);
		if (length - c >= TS_SIZE-4){
			l = write_ts_header(pid, counter, pes_start
					     , obuf, TS_SIZE-4);
			memcpy(obuf+l, p->buf+c, TS_SIZE-l);
			c += TS_SIZE-l;
		} else { 
			l = write_ts_header(pid, counter, pes_start
					     , obuf, length-c);
			memcpy(obuf+l, p->buf+c, TS_SIZE-l);
			c = length;
		}
		p->func(obuf,188,p->data);
		pes_start = 0;
	}
}


void setup_pes2ts( p2p *p, uint16_t pida, uint16_t pidv, 
		   void (*ts_write)(uint8_t *buf, int count, void *data),
		   void *data)
{
	init_p2p( p, ts_write, 2048);
	p->pida = pida;
	p->pidv = pidv;
	p->acounter = 0;
	p->vcounter = 0;
	p->count1 = 0;
	p->count0 = 0;
	p->data = data;
}

void pes_to_ts( p2p *p,uint8_t *buf ,int count )
{
	get_pes(buf,count, p,pes_in_ts);
}


void setup_ts2pes( ipack *pa, ipack *pv, uint16_t pida, uint16_t pidv, 
		   void (*pes_write)(uint8_t *buf, int count, void *data),
		   void *priv)
{
	init_ipack( pa, 2048, pes_write,0);
	init_ipack( pv, 2048, pes_write,0);
	pa->pid = pida;
	pv->pid = pidv;
	pa->data = priv;
	pv->data = priv;
}

void ts_to_pes( ipack *p, uint8_t *buf) // don't need count (=188)
{
	uint8_t off = 0;

	if (!buf || !p ){
		return;
	}


	if ( buf[1]&PAY_START) {
		if (p->plength == MMAX_PLENGTH-6 && p->found>6){
			p->plength = p->found-6;
			p->found = 0;
			send_ipack(p);
			reset_ipack(p);
		}
	}

	if ( buf[3] & ADAPT_FIELD) {  // adaptation field?
		off = buf[4] + 1;
		if (off+4 > 187) return;
	}

	instant_repack(buf+4+off, TS_SIZE-4-off, p );
}




unsigned int audioCapabilities(int fd)
{

	int ans;

	if ( ioctl(fd,AUDIO_GET_CAPABILITIES, &ans) < 0 ){
		perror("AUDIO GET CAPABILITIES: ");
		return -1;
	}

	return ans;
}


int audioSetStreamtype(int fd, char *format)
{

	int ans;
	unsigned int cap;
	unsigned int f = 0;

	cap = audioCapabilities(fd);

	if (!strcmp(format, "DTS"))   f = AUDIO_CAP_DTS;
	if (!strcmp(format, "LPCM"))  f = AUDIO_CAP_LPCM;
	if (!strcmp(format, "MP1"))   f = AUDIO_CAP_MP1;
	if (!strcmp(format, "MP2"))   f = AUDIO_CAP_MP2;
	if (!strcmp(format, "MP3"))   f = AUDIO_CAP_MP3;
	if (!strcmp(format, "AAC"))   f = AUDIO_CAP_AAC;
	if (!strcmp(format, "OGG"))   f = AUDIO_CAP_OGG;
	if (!strcmp(format, "SDDS"))  f = AUDIO_CAP_SDDS;
	if (!strcmp(format, "AC3"))   f = AUDIO_CAP_AC3;

	if (!(cap & f)) {
		fprintf(stderr,"Can't set audio format %s, not supported\n",format);
		return -1;
	}

	if ( ( ans = ioctl(fd,AUDIO_SET_STREAMTYPE, f)) < 0 ){
		perror("AUDIO SET STREAMTYPE: ");
		return -1;
	}

	return ans;
}

unsigned int videoCapabilities(int fd)
{

	int ans;

	if ( ioctl(fd,VIDEO_GET_CAPABILITIES, &ans) < 0 ){
		perror("VIDEO GET CAPABILITIES: ");
		return -1;
	}

	return ans;
}

int videoSetStreamtype(int fd, char *format)
{

	int ans;
	unsigned int cap;
	unsigned int f = 0;

	cap = videoCapabilities(fd);

	if (!strcmp(format, "MPEG1"))   f = VIDEO_CAP_MPEG1;
	if (!strcmp(format, "MPEG2"))  f = VIDEO_CAP_MPEG2;
	if (!strcmp(format, "SYS"))   f = VIDEO_CAP_SYS;
	if (!strcmp(format, "PROG"))   f = VIDEO_CAP_PROG;
	if (!strcmp(format, "SPU"))   f = VIDEO_CAP_SPU;
	if (!strcmp(format, "NAVI"))   f = VIDEO_CAP_NAVI;
	if (!strcmp(format, "CSS"))   f = VIDEO_CAP_CSS;

	if (!(cap & f)) {
		fprintf(stderr,"Can't set audio format %s, not supported\n",format);
		return -1;
	}

	if ( (ans = ioctl(fd,VIDEO_SET_STREAMTYPE, f)) < 0 ){
		perror("VIDEO SET STREAMTYPE: ");
		return -1;
	}

	return ans;
}


int videoStop(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_STOP,0) < 0)){
		perror("VIDEO STOP: ");
		return -1;
	}

	return 0;
}

int videoPlay(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_PLAY) < 0)){
		perror("VIDEO PLAY: ");
		return -1;
	}

	return 0;
}


int videoClear(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_CLEAR_BUFFER) < 0)){
		perror("VIDEO CLEAR BUFFER: ");
		return -1;
	}

	return 0;
}


int videoFreeze(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_FREEZE) < 0)){
		perror("VIDEO FREEZE: ");
		return -1;
	}

	return 0;
}


int videoContinue(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_CONTINUE) < 0)){
		perror("VIDEO CONTINUE: ");
		return -1;
	}

	return 0;
}

int videoSelectSource(int fd, video_stream_source_t source)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_SELECT_SOURCE, source) < 0)){
		perror("VIDEO SELECT SOURCE: ");
		return -1;
	}

	return 0;
}



int videoSetBlank(int fd, boolean state)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_SET_BLANK, state) < 0)){
		perror("VIDEO SET BLANK: ");
		return -1;
	}

	return 0;
}

int videoFastForward(int fd,int nframes)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_FAST_FORWARD, nframes) < 0)){
		perror("VIDEO FAST FORWARD: ");
		return -1;
	}

	return 0;
}

int videoSlowMotion(int fd,int nframes)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_SLOWMOTION, nframes) < 0)){
		perror("VIDEO SLOWMOTION: ");
		return -1;
	}

	return 0;
}

int videoGetStatus(int fd)
{
	struct video_status stat;
	int ans;

	if ( (ans = ioctl(fd,VIDEO_GET_STATUS, &stat) < 0)){
		perror("VIDEO GET STATUS: ");
		return -1;
	}

	printf("Video Status:\n");
	printf("  Blank State          : %s\n",
	       (stat.video_blank ? "BLANK" : "STILL"));
	printf("  Play State           : ");
	switch ((int)stat.play_state){
	case VIDEO_STOPPED:
		printf("STOPPED (%d)\n",stat.play_state);
		break;
	case VIDEO_PLAYING:
		printf("PLAYING (%d)\n",stat.play_state);
		break;
	case VIDEO_FREEZED:
		printf("FREEZED (%d)\n",stat.play_state);
		break;
	default:
		printf("unknown (%d)\n",stat.play_state);
		break;
	}
	
	printf("  Stream Source        : ");
	switch((int)stat.stream_source){
	case VIDEO_SOURCE_DEMUX:
		printf("DEMUX (%d)\n",stat.stream_source);
		break;
	case VIDEO_SOURCE_MEMORY:
		printf("MEMORY (%d)\n",stat.stream_source);
		break;
	default:
		printf("unknown (%d)\n",stat.stream_source);
		break;
	}

	printf("  Format (Aspect Ratio): ");
	switch((int)stat.video_format){
	case VIDEO_FORMAT_4_3:
		printf("4:3 (%d)\n",stat.video_format);
		break;
	case VIDEO_FORMAT_16_9:
		printf("16:9 (%d)\n",stat.video_format);
		break;
	default:
		printf("unknown (%d)\n",stat.video_format);
		break;
	}

	printf("  Display Format       : ");
	switch((int)stat.display_format){
	case VIDEO_PAN_SCAN:
		printf("Pan&Scan (%d)\n",stat.display_format);
		break;
	case VIDEO_LETTER_BOX:
		printf("Letterbox (%d)\n",stat.display_format);
		break;
	case VIDEO_CENTER_CUT_OUT:
		printf("Center cutout (%d)\n",stat.display_format);
		break;
	default:
		printf("unknown (%d)\n",stat.display_format);
		break;
	}
	return 0;
}

int videoStillPicture(int fd, struct video_still_picture *sp)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_STILLPICTURE, sp) < 0)){
		perror("VIDEO STILLPICTURE: ");
		return -1;
	}

	return 0;
}

int audioStop(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_STOP,0) < 0)){
		perror("AUDIO STOP: ");
		return -1;
	}

	return 0;
}

int audioPlay(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_PLAY) < 0)){
		perror("AUDIO PLAY: ");
		return -1;
	}

	return 0;
}

int audioClear(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_CLEAR_BUFFER) < 0)){
		perror("AUDIO CLEAR BUFFER: ");
		return -1;
	}

	return 0;
}


int audioPause(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_PAUSE) < 0)){
		perror("AUDIO PAUSE: ");
		return -1;
	}

	return 0;
}


int audioContinue(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_CONTINUE) < 0)){
		perror("AUDIO CONTINUE: ");
		return -1;
	}

	return 0;
}

int audioSelectSource(int fd, audio_stream_source_t source)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_SELECT_SOURCE, source) < 0)){
		perror("AUDIO SELECT SOURCE: ");
		return -1;
	}

	return 0;
}


int audioSetID(int fd,int id)
{
        int ans;

        if ( (ans = ioctl(fd,AUDIO_SET_ID, id) < 0)){
                perror("AUDIO SET ID ");
                return -1;
        }

        return 0;
}

int audioSetExtID(int fd,int id)
{
        int ans;

        if ( (ans = ioctl(fd,AUDIO_SET_EXT_ID, id) < 0)){
                perror("AUDIO SET EXT ID ");
                return -1;
        }

        return 0;
}

int audioSetMute(int fd, boolean state)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_SET_MUTE, state) < 0)){
		perror("AUDIO SET MUTE: ");
		return -1;
	}

	return 0;
}

int audioSetAVSync(int fd,boolean state)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_SET_AV_SYNC, state) < 0)){
		perror("AUDIO SET AV SYNC: ");
		return -1;
	}

	return 0;
}

int audioSetBypassMode(int fd,boolean mode)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_SET_BYPASS_MODE, mode) < 0)){
		perror("AUDIO SET BYPASS MODE: ");
		return -1;
	}

	return 0;
}


int audioChannelSelect(int fd, audio_channel_select_t select)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_CHANNEL_SELECT, select) < 0)){
		perror("AUDIO CHANNEL SELECT: ");
		return -1;
	}

	return 0;
}

int audioGetStatus(int fd)
{
	struct audio_status stat;
	int ans;

	if ( (ans = ioctl(fd,AUDIO_GET_STATUS, &stat) < 0)){
		perror("AUDIO GET STATUS: ");
		return -1;
	}

	printf("Audio Status:\n");
	printf("  Sync State          : %s\n",
	       (stat.AV_sync_state ? "SYNC" : "NO SYNC"));
	printf("  Mute State          : %s\n",
	       (stat.mute_state ? "muted" : "not muted"));
	printf("  Play State          : ");
	switch ((int)stat.play_state){
	case AUDIO_STOPPED:
		printf("STOPPED (%d)\n",stat.play_state);
		break;
	case AUDIO_PLAYING:
		printf("PLAYING (%d)\n",stat.play_state);
		break;
	case AUDIO_PAUSED:
		printf("PAUSED (%d)\n",stat.play_state);
		break;
	default:
		printf("unknown (%d)\n",stat.play_state);
		break;
	}
	
	printf("  Stream Source       : ");
	switch((int)stat.stream_source){
	case AUDIO_SOURCE_DEMUX:
		printf("DEMUX (%d)\n",stat.stream_source);
		break;
	case AUDIO_SOURCE_MEMORY:
		printf("MEMORY (%d)\n",stat.stream_source);
		break;
	default:
		printf("unknown (%d)\n",stat.stream_source);
		break;
	}

	printf("  Channel Select      : ");
	switch((int)stat.channel_select){
	case AUDIO_STEREO:
		printf("Stereo (%d)\n",stat.channel_select);
		break;
	case AUDIO_MONO_LEFT:
		printf("Mono left(%d)\n",stat.channel_select);
		break;
	case AUDIO_MONO_RIGHT:
		printf("Mono right (%d)\n",stat.channel_select);
		break;
	default:
		printf("unknown (%d)\n",stat.channel_select);
		break;
	}
	printf("  Bypass Mode         : %s\n",
	       (stat.bypass_mode ? "ON" : "OFF"));

	return 0;

}



void load_iframe(int filefd, int fd)
{
	struct stat st;
	struct video_still_picture sp;

	fstat(filefd, &st);
	
	sp.iFrame = (char *) malloc(st.st_size);
	sp.size = st.st_size;
	printf("I-frame size: %d\n", sp.size);
	
	if(!sp.iFrame) {
		printf("No memory for I-Frame\n");
		return;
	}

	printf("read: %d bytes\n",read(filefd,sp.iFrame,sp.size));
	videoStillPicture(fd,&sp);

	sleep(3);
	videoPlay(fd);
}

int open_av(int *fdv,int *fda, int dev, int init )
{
	char devnamea[80];
	char devnamev[80];
	
	sprintf(devnamea,"/dev/dvb/adapter%d/audio0",dev);
	sprintf(devnamev,"/dev/dvb/adapter%d/video0",dev);
	
	if((*fdv = open(devnamev,O_RDWR|O_NONBLOCK)) < 0){
		return -1;
	}
	    
	if((*fda = open(devnamea,O_RDWR|O_NONBLOCK)) < 0){
		return -2;
	}
	
	if (!init) return 0;
	if (videoSetBlank(*fdv,false) < 0) return -3;
	if (audioSelectSource(*fda,AUDIO_SOURCE_MEMORY) < 0) return -3;
	if (audioSetAVSync(*fda, true) < 0) return -3;
	if (audioPlay(*fda) < 0) return -3;
	return 0;
}



void set_pat_filt(int fd)
{
	struct dmx_sct_filter_params sctFilterParams;


	memset(&sctFilterParams.filter, 0, sizeof(struct dmx_filter));
	sctFilterParams.pid                       = 0;
	sctFilterParams.filter.filter[0]          = 0x00;
	sctFilterParams.filter.mask[0]            = 0x00;
	sctFilterParams.timeout                   = 1000;
	sctFilterParams.flags                     = DMX_IMMEDIATE_START;

	if (ioctl(fd, DMX_SET_FILTER, &sctFilterParams) < 0)  
		perror("DMX SET PAT FILTER:");

}

void set_pmt_filt(int fd,uint16_t ppid)
{
	struct dmx_sct_filter_params sctFilterParams;


	memset(&sctFilterParams.filter, 0, sizeof(struct dmx_filter));
	sctFilterParams.pid                       = ppid;
	sctFilterParams.filter.filter[0]          = 0x02;
	sctFilterParams.filter.mask[0]            = 0xFF;
	sctFilterParams.timeout                   = 1000;
	sctFilterParams.flags                     = DMX_IMMEDIATE_START;

	if (ioctl(fd, DMX_SET_FILTER, &sctFilterParams) < 0)  
		perror("DMX SET PMT FILTER:");

}

void set_av_filts(int vfd,int afd,uint16_t vpid,uint16_t apid)
{
	struct dmx_pes_filter_params pesFilterParams; 
	static int bset=0;

	if (!bset){
		if (ioctl(vfd, DMX_SET_BUFFER_SIZE, 64*1024) < 0)  
			perror("DMX SET BUFFER:");
		if (ioctl(afd, DMX_SET_BUFFER_SIZE, 64*1024) < 0) 
			perror("DMX SET BUFFER:");
		bset = 1;
	} else {
		ioctl(vfd, DMX_STOP, 0);
		ioctl(afd, DMX_STOP, 0);
	}

	pesFilterParams.pid = vpid; 
	pesFilterParams.input = DMX_IN_DVR; 
	pesFilterParams.output = DMX_OUT_DECODER; 
	pesFilterParams.pes_type = DMX_PES_VIDEO; 
	pesFilterParams.flags = DMX_IMMEDIATE_START;
	if (ioctl(vfd, DMX_SET_PES_FILTER, &pesFilterParams) < 0) 
		perror("DMX SET VIDEO FILTER:");
  
	pesFilterParams.pid = apid;
	pesFilterParams.input = DMX_IN_DVR; 
	pesFilterParams.output = DMX_OUT_DECODER; 
	pesFilterParams.pes_type = DMX_PES_AUDIO; 
	pesFilterParams.flags = DMX_IMMEDIATE_START;
  
	if (ioctl(afd, DMX_SET_PES_FILTER, &pesFilterParams) < 0)
		perror("DMX SET AUDIO FILTER:");
}

void stop_av_filts(int afd,int vfd)
{
	if (ioctl(vfd, DMX_STOP) < 0)
		perror("DMX STOP");
  
	if (ioctl(afd, DMX_STOP) < 0)
		perror("DMX STOP");
}


uint16_t get_pmt_pid(int fd)
{
	u_char sec[MAX_SECTION_SIZE];
	int len, i;
	uint16_t cpid = 0;
	uint16_t length;
	len=read(fd, sec, 4096);
	
	if (len <= 0) return 0;
	
	length  = (sec[1]& 0x0F)<<8;
	length |= (sec[2]& 0xFF);
	
	for (i = 8; i< length-1 && cpid == 0; i+=4){
		if (sec[i] != 0 || sec[i+1] !=0){
			cpid = get_pid(sec+i+2);
//			printf("TS: PMT PID: %04x\n",cpid);
		}
	}
	return cpid;
}


void get_av_pids(int fd, uint16_t *vpid, uint16_t *apid)
{		
	u_char sec[MAX_SECTION_SIZE];
	int len, i, ilength;
	uint16_t length;

	len = read(fd, sec, MAX_SECTION_SIZE);
	 
	if (len <= 0) return;

	length  = (sec[1]& 0x0F)<<8;
	length |= (sec[2]& 0xFF);
        
	ilength = (unsigned short)
		((sec[10]&3)<<8);
	ilength |= (sec[11]&0xFF);
	for (i = 12+ilength; i< length-1; i+=5){
		if (sec[i] == 0x02){
			*vpid = get_pid(sec+i+1);
//			printf("TS: VIDEO PID: %d\n",*vpid);
		}
		if (sec[i] == 0x03 || sec[i] == 0x04){
			*apid = get_pid(sec+i+1);
//			printf("TS: AUDIO PID: %d\n",*apid);
		}
		if (*vpid && *apid) break;
		i+=((sec[i+3]&15)<<8)|sec[i+4];
	}
}

int new_get_av_pids(int fd, uint16_t *vpid, uint16_t *apids, int maxapids)
{		
	u_char sec[MAX_SECTION_SIZE];
	int len, i;
	uint16_t pcrpid;
	int slen, ilen, c;
	time_t count = time(0)+10;
	int apidnum = 0;
	uint16_t silen;
	uint8_t maxsec = 0;
	uint8_t secs = 0;


	while (secs <= maxsec && count > time(0)) {
		if ( (len = read(fd, sec, MAX_SECTION_SIZE))>0 ){
			maxsec=sec[7];
			secs++;
			slen = (((sec[1]&0x0f) << 8) | sec[2]) +3;
			ilen = ((sec[10]&0x03)  <<8) | sec[11];
			pcrpid =  get_pid(sec+8);
			c = 12+ilen;
			while (c < slen){
				switch (sec[c]) {
				case 1: 
				case 2: 
					*vpid = get_pid(sec+c+1);
					break;
				case 3: 
				case 4: 
				{
					int afound = 0;
					uint16_t apid = get_pid(sec+c+1); 

					if (apidnum >= maxapids) {
						return maxapids;
						break;
					}
					
					for (i = 0; i < apidnum; i++){
						if ( apid == apids[i]){
							afound = 1;
							break;
						}
					}
					if (! afound){
						apids[apidnum++] = apid;
					}
					break; 
				}
				default:
					break;
				}
				silen = (((sec[c+3] & 0x0f) << 8) | sec[c+4]);
				c += 5 + silen;
			}
		}
	}
	return apidnum;
}


int seek_mpg_start(uint8_t *buf, int size)
{
        int found = 0;
        int c=0;
        int seq = 0;
        int mpeg = 0;
        int mark = 0;

        while ( !seq ){
                while (found != 4){
                        switch (found) {
                        case 0:
                                if ( buf[c] == 0x00 ) found++;
                                c++;
                                break;
                        case 1:
                                if ( buf[c] == 0x00 ) found++;
                                else found = 0;
                                c++;
                                break;                     
                        case 2:
                                if ( buf[c] == 0x01 ) found++;
                                else found = 0;
                                if ( buf[c] == 0x00 ) found = 2;
                                c++;
                                break;
                        
                        case 3:
                                if ( (buf[c] & 0xe0) == 0xe0 ) found++;
                                else found = 0;
                                c++;
                                break;
                        }
                        if (c >= size) return -1;
                }
                
                if (found == 4){
                        mark = c-4;
                        c+=2;
                        if (c >= size) return -1;

                        if ( (buf[c] & 0xC0) == 0x80 ){
                                mpeg = 2;
                                c += 2;
                                if (c >= size) return -1;
                                c += buf[c]+1;
                                if (c >= size) return -1;
                        } else {
                                mpeg = 1;
                                while( buf[c] == 0xFF ) {
                                        c++;
                                        if (c >= size) return -1;
                                }
                                if ( (buf[c] & 0xC0) == 0x40) c+=2;
                                if (c >= size) return -1;
                                if ( (buf[c] & 0x30) ){
                                        if ( (buf[c] & 0x30) == 0x20) c+=5;
                                        else c+=10;
                                } else c++;
                                if (c >= size) return -1;
                        }

                        if ( buf[c] == 0x00 && 
                             buf[c+1] == 0x00 && 
                             buf[c+2] == 0x01 && 
                             buf[c+3] == 0xB3 ) 
                                seq = 1;
                }
                found = 0;
        }

        return size-mark;
}

#define CHECKBUF (1024*1024)
#define SKIP (10*1024*1024)
int64_t next_gop(int fd, int mb)
{
	uint8_t buf[CHECKBUF];	
	int mark;
	off_t c,n;

	c = lseek(fd, 0, SEEK_CUR);
	n = lseek(fd, mb*SKIP, SEEK_CUR);

	read(fd, buf, CHECKBUF);
	if ( (mark = seek_mpg_start(buf,CHECKBUF)) < 0){
		//fprintf(stderr,"Couldn't find sequence header\n");
		mark = 0;
	}
	
	c = lseek(fd, n+mark, SEEK_SET);

	return c;
}

int64_t prev_gop(int fd, int mb)
{
	uint8_t buf[CHECKBUF];	
	int mark;
	off_t c,n;
	off_t back;

	c = lseek(fd, 0, SEEK_CUR);

	if (mb*SKIP > c) back = 0;
	else back = c - mb*SKIP;

	n = lseek(fd, back, SEEK_SET);

	read(fd, buf, CHECKBUF);
	if ( (mark = seek_mpg_start(buf,CHECKBUF)) < 0){
		//fprintf(stderr,"Couldn't find sequence header\n");
		mark = 0;
	}
	
	c = lseek(fd, n+mark, SEEK_SET);

	return c;
}

#define CHECKTS (3*188)
int64_t goto_ts_start(int fd, off_t cur)
{
	int64_t c;
	uint8_t buf[CHECKTS];
	int tsc;
	int found = 0;

	c = cur - 187;
	if (c < 0) c = 0;
	lseek(fd, c , SEEK_SET);

	read(fd, buf, CHECKTS);
	tsc = 0;
	while (!found && tsc < CHECKTS-188){
		if (buf[tsc] == 0x47 && buf[tsc+188] == 0x47) found = 1;
		else tsc++;
	}

	c = lseek(fd, c+tsc, SEEK_SET);

	return c;
}

int64_t next_gop_ts(int fd, int mb)
{
	uint8_t buf[CHECKBUF];	
	int mark;
	off_t c,n;

	c = lseek(fd, 0, SEEK_CUR);
	n = lseek(fd, mb*SKIP, SEEK_CUR);

	read(fd, buf, CHECKBUF);
	if ( (mark = seek_mpg_start(buf,CHECKBUF)) < 0){
		//fprintf(stderr,"Couldn't find sequence header\n");
		mark = 0;
	} 

	c = goto_ts_start(fd, n+mark);

	return c;
}


int64_t prev_gop_ts(int fd, int mb)
{
	uint8_t buf[CHECKBUF];	
	int mark;
	off_t c,n;
	off_t back;

	c = lseek(fd, 0, SEEK_CUR);

	if (mb*SKIP > c) back = 0;
	else back = c - mb*SKIP;

	n = lseek(fd, back, SEEK_SET);

	read(fd, buf, CHECKBUF);
	if ( (mark = seek_mpg_start(buf,CHECKBUF)) < 0){
		//fprintf(stderr,"Couldn't find sequence header\n");
		mark = 0;
	}
	
	c = goto_ts_start(fd, n+mark);

	return c;
}


static inline uint8_t *skip_pes_header(uint8_t **bufp)
{
	uint8_t *inbuf = *bufp;
	uint8_t *buf = inbuf;
	uint8_t *pts = NULL;
	int skip = 0;

	static int mpeg1_skip_table[16] = {
	  1, 0xffff,      5,     10, 0xffff, 0xffff, 0xffff, 0xffff,
	  0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff
	};



	if ((inbuf[6] & 0xc0) == 0x80){	/* mpeg2 */
		if (buf[7] & PTS_ONLY)
			pts = buf+9;
		else pts = NULL;
		buf = inbuf + 9 + inbuf[8];
	} else {	/* mpeg1 */
		for (buf = inbuf + 6; *buf == 0xff; buf++)
			if (buf == inbuf + 6 + 16) {
				fprintf (stderr, "too much stuffing\n");
				break;
			}
		if ((*buf & 0xc0) == 0x40)
			buf += 2;
		skip = mpeg1_skip_table [*buf >> 4];
		if (skip == 5 || skip == 10) pts = buf;
		else pts = NULL;

		buf += mpeg1_skip_table [*buf >> 4];
	}

	*bufp = buf;
	return pts;
}

#ifdef HAVE_LIBA52

/* taken from audio_out_oss.c by Aaron Holtzman */
static inline int16_t convert (int32_t i)
{
    if (i > 0x43c07fff)
	return 32767;
    else if (i < 0x43bf8000)
	return -32768;
    else
	return i - 0x43c00000;
}

/* taken from audio_out_oss.c by Aaron Holtzman */
static inline int float_to_int (float * _f, int8_t * s8, int flags)
{
    int i;
    int32_t * f = (int32_t *) _f;
    int chans=0;
    int16_t *s16 = (int16_t *) s8;

    switch (flags) {
    case A52_MONO:
	for (i = 0; i < 256; i++) {
	    s16[5*i] = s16[5*i+1] = s16[5*i+2] = s16[5*i+3] = 0;
	    s16[5*i+4] = convert (f[i]);
	}
	chans = 2;
	break;
    case A52_CHANNEL:
    case A52_STEREO:
    case A52_DOLBY:
	for (i = 0; i < 256; i++) {
	    s16[2*i] = convert (f[i]);
	    s16[2*i+1] = convert (f[i+256]);
	}
	chans = 2;
	break;
    case A52_3F:
	for (i = 0; i < 256; i++) {
	    s16[5*i] = convert (f[i]);
	    s16[5*i+1] = convert (f[i+512]);
	    s16[5*i+2] = s16[5*i+3] = 0;
	    s16[5*i+4] = convert (f[i+256]);
	}
	chans = 5;
	break;
    case A52_2F2R:
	for (i = 0; i < 256; i++) {
	    s16[4*i] = convert (f[i]);
	    s16[4*i+1] = convert (f[i+256]);
	    s16[4*i+2] = convert (f[i+512]);
	    s16[4*i+3] = convert (f[i+768]);
	}
	chans = 4;
	break;
    case A52_3F2R:
	for (i = 0; i < 256; i++) {
	    s16[5*i] = convert (f[i]);
	    s16[5*i+1] = convert (f[i+512]);
	    s16[5*i+2] = convert (f[i+768]);
	    s16[5*i+3] = convert (f[i+1024]);
	    s16[5*i+4] = convert (f[i+256]);
	}
	chans = 5;
	break;
    case A52_MONO | A52_LFE:
	for (i = 0; i < 256; i++) {
	    s16[6*i] = s16[6*i+1] = s16[6*i+2] = s16[6*i+3] = 0;
	    s16[6*i+4] = convert (f[i+256]);
	    s16[6*i+5] = convert (f[i]);
	}
	chans = 6;
	break;
    case A52_CHANNEL | A52_LFE:
    case A52_STEREO | A52_LFE:
    case A52_DOLBY | A52_LFE:
	for (i = 0; i < 256; i++) {
	    s16[6*i] = convert (f[i+256]);
	    s16[6*i+1] = convert (f[i+512]);
	    s16[6*i+2] = s16[6*i+3] = s16[6*i+4] = 0;
	    s16[6*i+5] = convert (f[i]);
	}
	chans = 6;
	break;
    case A52_3F | A52_LFE:
	for (i = 0; i < 256; i++) {
	    s16[6*i] = convert (f[i+256]);
	    s16[6*i+1] = convert (f[i+768]);
	    s16[6*i+2] = s16[6*i+3] = 0;
	    s16[6*i+4] = convert (f[i+512]);
	    s16[6*i+5] = convert (f[i]);
	}
	chans = 6;
	break;
    case A52_2F2R | A52_LFE:
	for (i = 0; i < 256; i++) {
	    s16[6*i] = convert (f[i+256]);
	    s16[6*i+1] = convert (f[i+512]);
	    s16[6*i+2] = convert (f[i+768]);
	    s16[6*i+3] = convert (f[i+1024]);
	    s16[6*i+4] = 0;
	    s16[6*i+5] = convert (f[i]);
	}
	chans = 6;
	break;
    case A52_3F2R | A52_LFE:
	for (i = 0; i < 256; i++) {
	    s16[6*i] = convert (f[i+256]);
	    s16[6*i+1] = convert (f[i+768]);
	    s16[6*i+2] = convert (f[i+1024]);
	    s16[6*i+3] = convert (f[i+1280]);
	    s16[6*i+4] = convert (f[i+512]);
	    s16[6*i+5] = convert (f[i]);
	}
	chans = 6;
	break;
    }

    return 2*chans*256;

}





/* taken from a52dec.c by Aaron Holtzman */
uint8_t a52_buf[3840];
uint8_t * a52_bufptr = a52_buf;
uint8_t * a52_bufpos = a52_buf + 7;

void init_a52_buf()
{
	a52_bufptr = a52_buf;
	a52_bufpos = a52_buf + 7;
}

void a52_decode_data (int fd, uint8_t *pts, uint8_t * start, 
		      uint8_t * end, a52_state_t *state)
{
	uint8_t intsamples[6*256*4];
	int len=0;
        uint16_t *s=(uint16_t *)intsamples;
    /*
     * sample_rate and flags are static because this routine could
     * exit between the a52_syncinfo() and the ao_setup(), and we want
     * to have the same values when we get back !
     */

	static int sample_rate;
	static int flags;
	int bit_rate;

	while (start < end) {
		*a52_bufptr++ = *start++;
		if (a52_bufptr == a52_bufpos) {
			if (a52_bufpos == a52_buf + 7) {
				int length;
				
				length = a52_syncinfo (a52_buf, &flags, 
						       &sample_rate, 
						       &bit_rate);
				if (!length) {
					for (a52_bufptr = a52_buf; a52_bufptr < a52_buf + 6; 
					     a52_bufptr++)
						a52_bufptr[0] = a52_bufptr[1];
					continue;
				}
				a52_bufpos = a52_buf + length;
			} else {
				sample_t level, bias;
				int i;

				flags = A52_DOLBY;
				level = 1;
				bias = 384;

				flags |= A52_ADJUST_LEVEL;
				if (a52_frame (state, a52_buf, &flags, &level, 
					       bias))
					goto error;
				for (i = 0; i < 6; i++) {
					if (a52_block (state))
						goto error;
					len += float_to_int (
						a52_samples(state), 
						intsamples+len,
						flags);
				}
		
				for(i=0;i<len/2;i++) s[i]=htons(s[i]); 
				pcm_play (fd, intsamples, sample_rate, 
					  pts, len);
				len = 0;
			
				a52_bufptr = a52_buf;
				a52_bufpos = a52_buf + 7;
				continue;
			error:
				a52_bufptr = a52_buf;
				a52_bufpos = a52_buf + 7;
			}
		}
	}
}

#define BUFFER_SIZE 262144
static uint8_t buffer[BUFFER_SIZE];

void ps_loop (int fd, uint8_t *inbuf, int len, 
	      a52_state_t *state, int a52_track)
{
	static uint8_t * buf = NULL;
	static uint8_t * end;
	uint8_t * tmp1;
	uint8_t * tmp2;
	uint8_t *mbuf = inbuf;
	int mlen;
	uint8_t *pts = NULL;

	if (!buf) buf = buffer;
	do {
		if (len >= buffer + BUFFER_SIZE - buf) {
			mlen = buffer + BUFFER_SIZE - buf; 
			memcpy (buf, mbuf,  mlen); 
			end = buf + mlen; 
			len -= mlen; 
			mbuf += mlen; 
		}  else {
			memcpy (buf, mbuf, len); 
			end = buf + len;
			len = 0;
		}
		buf = buffer;
	    
		while (buf + 4 <= end) {
			tmp2 = buf + 6 + (buf[4] << 8) + buf[5];
			if (tmp2 > end)
				goto copy;
			

			tmp1 = buf;
			pts = skip_pes_header(&tmp1);
/*
			if ((buf[6] & 0xc0) == 0x80){	
				if (buf[7] & PTS_ONLY)
					pts = buf+9;
				else pts = NULL;
				tmp1 = buf + 9 + buf[8];
			} else {
				int skip;
				
				for (tmp1 = buf + 6; *tmp1 == 0xff; tmp1++)
					if (tmp1 == buf + 6 + 16) {
						fprintf (stderr, "too much stuffing\n");
						buf = tmp2;
						break;
					}
				if ((*tmp1 & 0xc0) == 0x40)
					tmp1 += 2;
				skip = mpeg1_skip_table [*tmp1 >> 4];
				if (skip == 2 || skip == 3) pts = tmp1;
				else pts = NULL;

				tmp1 += skip;
			}
*/
			if (*tmp1 == a52_track) {	/* a52 */
				tmp1 += 4;
				if (tmp1 < tmp2)
					a52_decode_data (fd, pts, tmp1, tmp2, 
							 state);
			}
			buf = tmp2;
			break;
		}
	    
		if (buf < end) {
		copy:
			/* we only pass here for mpeg1 ps streams */
			memmove (buffer, buf, end - buf);
		}
		buf = buffer + (end - buf);
		
	} while (end == buffer + BUFFER_SIZE);
}

#endif

#ifdef HAVE_LIBMAD
static struct mad_stream mad_stream;
static struct mad_frame  mad_frame;
static struct mad_synth  mad_synth;
#define MAD_LEN  16384
#define MAD_SAMPLES 8192
uint8_t mad_buffer[MAD_LEN];
uint16_t mad_samples[MAD_SAMPLES];
int mad_full;

void init_mad(){
	mad_frame_init(&mad_frame);
	mad_stream_init(&mad_stream);
	mad_synth_init(&mad_synth);
	mad_full = 0;
}

void close_mad () {

	mad_frame_finish (&mad_frame);
	mad_stream_finish(&mad_stream);
	mad_synth_finish (&mad_synth);

}

static inline uint16_t scale(mad_fixed_t sample)
{
	/* round */
	sample += (1L << (MAD_F_FRACBITS - 16));
	
	/* clip */
	if (sample >= MAD_F_ONE)
		sample = MAD_F_ONE - 1;
	else if (sample < -MAD_F_ONE)
		sample = -MAD_F_ONE;
	
	/* quantize */
	return ntohs(sample >> (MAD_F_FRACBITS + 1 - 16));
}


void mad_decode(int fd, uint8_t *inbuf, int buf_size) 
{
	uint8_t *pts = NULL;
	uint8_t *buf = inbuf;
	int rate;
	
	pts = skip_pes_header(&buf);
	buf_size -= buf - inbuf;

	if (buf_size>(MAD_LEN-mad_full)) {
		printf ("libmad: ALERT input buffer too small (%d bytes, %d avail)!\n",
			buf_size, MAD_LEN-mad_full);
		buf_size = MAD_LEN-mad_full;
	}
	
	memcpy (&mad_buffer[mad_full], buf, buf_size);
	mad_full += buf_size;

	mad_stream_buffer (&mad_stream, mad_buffer, mad_full);

	while (1) {

		if (mad_frame_decode (&mad_frame, &mad_stream) != 0) {
			
			if (mad_stream.next_frame) {
				int num_bytes = mad_buffer + mad_full - 
					mad_stream.next_frame;
          
				memmove(mad_buffer, mad_stream.next_frame, 
					num_bytes);
				mad_full = num_bytes;
			}

			switch (mad_stream.error) {

			case MAD_ERROR_BUFLEN:
				return;

			default: 
				mad_stream_buffer (&mad_stream, mad_buffer, 
						   mad_full);
			}

		}

		mad_synth_frame (&mad_synth, &mad_frame);
		rate = mad_frame.header.samplerate;

		if (rate){
			unsigned int nchannels, nsamples;
			struct mad_pcm *pcm = &mad_synth.pcm;
			uint16_t *output = mad_samples;

			int i;

			nchannels = pcm->channels;
			nsamples  = pcm->length;
			
			for(i=0; i<mad_synth.pcm.length; ++i) {
				*output++ = scale(pcm->samples[0][i]);
				*output++ = scale(pcm->samples[0][i]);
			}			
			pcm_play( fd, (uint8_t *)mad_samples, 
				  rate, pts, pcm->length*4);

			pts = NULL;
		} 
	}
} 
#endif

void find_avpids(int fd, uint16_t *vpid, uint16_t *apid)
{
	uint8_t buf[IN_SIZE];
	int count;
	int i;	
	int off =0;

	while ( *apid == 0 || *vpid == 0){
		count = read(fd, buf, IN_SIZE);
		for (i = 0; i < count-7; i++){
			if (buf[i] == 0x47){
				if (buf[i+1] & 0x40){
					off = 0;
					if ( buf[3+i] & 0x20)//adapt field?
						off = buf[4+i] + 1;
					switch(buf[i+7+off]){
					case VIDEO_STREAM_S ... VIDEO_STREAM_E:
						*vpid = get_pid(buf+i+1);
						break;
					case PRIVATE_STREAM1:
					case AUDIO_STREAM_S ... AUDIO_STREAM_E:
						*apid = get_pid(buf+i+1);
						break;
					}
				}
				i += 187;
			}
			if (*apid != 0 && *vpid != 0) break;
		}
	}
}

void find_bavpids(uint8_t *buf, int count, uint16_t *vpid, uint16_t *apid)
{
	int i;	
	int founda = 0;
	int foundb = 0;
	int off = 0;
	
	*vpid = 0;
	*apid = 0;
	for (i = 0; i < count-7; i++){
		if (buf[i] == 0x47){
			if ((buf[i+1] & 0xF0) == 0x40){
				off = 0;
				if ( buf[3+i] & 0x20)  // adaptation field?
					off = buf[4+i] + 1;
				
				if (buf[off+i+4] == 0x00 && 
				    buf[off+i+5] == 0x00 &&
				    buf[off+i+6] == 0x01){
					switch(buf[off+i+7]){
					case VIDEO_STREAM_S ... VIDEO_STREAM_E:
						*vpid = get_pid(buf+i+1);
						foundb=1;
						break;
					case PRIVATE_STREAM1:
					case AUDIO_STREAM_S ... AUDIO_STREAM_E:
						*apid = get_pid(buf+i+1);
						founda=1;
						break;
					}
				}
			}
			i += 187;
		}
		if (founda && foundb) break;
	}
}

void set_volume(int fda, int vol, av_settings *av)
{
	int ans;
	audio_mixer_t am;
	static int novol = 0;

	if (novol) return;
	if (vol == av->volume) return;

	if (vol > 255) vol = 255;
	if (vol < 0  ) vol = 0;
	av->volume = vol;
	am.volume_left = vol;
	am.volume_right = vol;

	if ( (ans = ioctl(fda,AUDIO_SET_MIXER, &am) < 0)){
		novol = 1;
		return;
	}
}

void mute(int fda, int on, av_settings *av)
{
	static int vol = 0;
	static int is_muted = 0;

	if (!on && is_muted){
		if (!vol) vol = 200;
		set_volume(fda, vol, av);
		is_muted = 0;
	} 
	if (on && !is_muted){
		is_muted = 1;
		vol = av->volume;
		set_volume(fda, 0, av);
	}
}

void init_av(av_settings *av)
{
	av->volume = 200;
	av->a52_track = 0x80;
	av->mpx_track = AUDIO_STREAM_S;
	av->aud_ext = 0;
		
#ifdef HAVE_LIBA52
	av->state=NULL;
	av->soft_ac3 = 0;
#endif
#ifdef HAVE_LIBMAD
	av->soft_mpx = 0;
#endif

	av->needs_demux = 0;
	av->nopat = 0;
}



int av_switches(int cc, switch_t *sw, uint64_t *l){
	static int stopped = 0;
	static int trick = 0;
	int s;
	int fd = sw->fdv;
	int fda = sw->fda;
	int filefd = sw->filefd;
	int ts = sw->ts;
	av_settings *av = sw->av;
	
	if (fd < 0 || fda < 0) return -1;

	switch(cc){
	case 'z':
		mute(fda, 1, av);
		audioPause(fda);
		videoFreeze(fd);
		stopped = 1;
		trick = 1;
		return 4;
		break;
	case 's':
		videoStop(fd);
		mute(fda, 1, av);
		audioStop(fda);
		stopped = 1;
		return 4;
		break;
		
	case 'c':
		audioContinue(fda);
		videoContinue(fd);
		audioSetAVSync(fda, true);
		stopped = 0;
		trick = 0;
#ifdef HAVE_LIBA52
		if (av->soft_ac3)  init_a52_buf();
#endif
		mute(fda, 0, av);
		return 3;
		break;
		
	case 'p':
		if (trick){
			audioContinue(fda);
			videoContinue(fd);
		} else {
			audioPlay(fda);
			videoPlay(fd);
		}
		trick = 0;
		stopped = 0;
		audioSetAVSync(fda, true);
		mute(fda, 0, av);
		return 3;
		break;
		
	case 'f':
		audioSetAVSync(fda, false);
		mute(fda, 1, av);
		audioPause(fda);
		videoFastForward(fd,0);
		stopped = 0;
		trick = 1;
		return 3;
		break;
		
	case 'm':
		audioSetAVSync(fda, false);
		mute(fda, 1, av);
		audioPause(fda);
		videoSlowMotion(fd,2);
		stopped = 0;
		trick = 1;
		return 3;
		break;
		
	case '+':
		set_volume(fda, av->volume+1, av);
		break;
		
	case '-':
		set_volume(fda, av->volume-1, av);
		break;
		
	case 'n':
		videoClear(fd);
		audioClear(fda);
		return 0;
		break;
		
	case 'b':
		videoClear(fd);
		audioClear(fda);
		return 1;
		break;
		
	case '0':
	case '1' ... '9':
		if (cc == '0') s = 10; 
		else s = 0x80 + cc - '1';
		if (s>0x87 || s < 0x80)
			s=0x80;
		av->a52_track = s;
		
		if (cc == '0') s = 10; 
		else s = AUDIO_STREAM_S + cc - '1';
		if (s>AUDIO_STREAM_E || s < AUDIO_STREAM_S)
			s=AUDIO_STREAM_S;
		av->mpx_track = s;

#ifdef HAVE_LIBA52
		if (av->soft_ac3)
			init_a52_buf();
		if (av->aud_ext && !av->soft_ac3){
#endif
			audioSetExtID(fda, av->a52_track);
			audioClear(fda);
#ifdef HAVE_LIBA52
		}
#endif
		break;
		
	case 'q':
		videoContinue(fd);
		exit(0);
		break;
		
	case 258:
		if (ts) *l = prev_gop_ts(filefd, 10);
		else *l = prev_gop(filefd, 10);
		return 2;
		break;
		
	case 260:
		if (ts) *l = prev_gop_ts(filefd, 1);
		else *l = prev_gop(filefd, 1);
		return 2;
		break;
		
	case 259:
		if (ts) *l = next_gop_ts(filefd, 10);
		else *l = next_gop(filefd, 10);
		return 2;
		break;
		
	case 261:
		if (ts) *l = next_gop_ts(filefd, 1);
		else *l = next_gop(filefd, 1);
		return 2;
		break;
	}
	return -1;
}



ssize_t write_demux(int fd, uint8_t *buf, int count, av_settings *av,
		    ssize_t (*w)(int fd, uint8_t *buf, int count))
{
	switch(buf[3]){
	case VIDEO_STREAM_S ... VIDEO_STREAM_E:
		return w(fd, buf, count);
		break;
	case AUDIO_STREAM_S ... AUDIO_STREAM_E:			
#ifdef HAVE_LIBMAD
		if (av->soft_mpx && av->mpx_track == buf[3]){
			mad_decode(fd, buf, count);
			return count;
		} else return w(fd, buf, count);
#else 
		return w(fd, buf, count);
#endif
		break;
	case PRIVATE_STREAM1:
#ifdef HAVE_LIBA52
		if (av->soft_ac3){
			ps_loop (fd, buf, count, 
				 av->state, av->a52_track);
			return count;
		} else return w(fd, buf, count);
#else 
		return w(fd, buf, count);
#endif
		break;
		
	default:
		return count;
	} 
}

void write_out_demux(uint8_t *buf, int count, void *priv)
{
	ipack *p = (ipack *) priv;
	av_settings *av = p->av;

	switch(buf[3]){

	case VIDEO_STREAM_S ... VIDEO_STREAM_E:
		write_demux(p->fd1, buf, count, av, poll_write);
		break;
	case AUDIO_STREAM_S ... AUDIO_STREAM_E:			
	case PRIVATE_STREAM1:
		write_demux(p->fd2, buf, count, av, poll_write);
		break;
		
	default:
		return;
	} 
}

#define IPACKS 2048
void play_file_video_mpg(switch_t *sw)
{
	char buf[BUFFY];
	int count, c;
	int written;
	struct pollfd pfd[1];
	int stop = 0;
	ipack pack;
	int fd = sw->fdv;
	int fda = sw->fda;
	int filefd = sw->filefd;
	av_settings *av = sw->av;

	pfd[0].fd = fd;
	pfd[0].events = POLLOUT;

	if (videoSetBlank(fd,false) < 0) return;
	if (videoSelectSource(fd,VIDEO_SOURCE_MEMORY) < 0) return;
	if (videoPlay(fd) < 0) return;
	

	if (fda) {
		if (audioSelectSource(fda,AUDIO_SOURCE_MEMORY) < 0) return;
		if (audioSetAVSync(fda, true) < 0) return;
		if (audioPlay(fda) < 0) return;
	}


	if (av->needs_demux){
		init_ipack(&pack, IPACKS, write_out_demux,0);
		pack.data = (void *)&pack;
		pack.fd   = fd;
		pack.fd1  = fd;
		pack.fd2  = fda;
		pack.av = av;
	}

	written = 0;
	count = 0;
	stop = 0;
	while ( stop < 10 ){
		if (written == count){
			if ( (count =  read(filefd, buf, BUFFY)) <= 0){
				stop++;
			}
			written = 0;
		}
		if (av->needs_demux){
			instant_repack(buf, count, &pack);
			written = count;
		}
		if (poll(pfd,1,100)){
			if ( !av->needs_demux && pfd[0].revents & POLLOUT){
				if ( written < count){
					c = write(fd,buf+written,
						  count-written);
					if (c > 0) written += c;
				} 
				
			}
		}

		if (sw->stop && *sw->stop)
			stop = 10;
	}
	if (sw->stop) *sw->stop = 1;
	audioSelectSource(fda,AUDIO_SOURCE_DEMUX);
	videoSelectSource(fd,VIDEO_SOURCE_DEMUX);
	videoPlay(fd);
	audioSetAVSync(fda, true);
}
	

#define MAXAPIDS 32
void play_file_dvr_ts(int fd_dvr, uint16_t apid, uint16_t vpid, int dev,
		      switch_t *sw)
{
	char buf[BUFFY2];
	int count = 0, c;
	int written;
	uint16_t ppid=0;
	int fd_vdemux;
	int fd_ademux;
	int fd;
	uint16_t apid_n;
	uint16_t vpid_n;
	char devnamed[80];
	uint16_t apids[MAXAPIDS];
	int napid;
	int set;
	int fdv = sw->fdv;
	int fda = sw->fda;
	int filefd = sw->filefd;
	av_settings *av = sw->av;


	sprintf(devnamed,"/dev/dvb/adapter%d/demux0",dev);

	if((fd = open(devnamed,O_RDWR|O_NONBLOCK)) < 0){
		perror(devnamed);
		return;
	}

	if (av->nopat) find_avpids(STDIN_FILENO, &vpid, &apid);

	if (apid && vpid) ppid = 1;
	else {
		apid = 0;
		vpid = 0;
		ppid = 0;
	}

	if ((fd_vdemux = open(devnamed, O_RDWR|O_NONBLOCK)) < 0){
		perror("DMX VIDEO open");
		return;
	}
	fprintf(stderr,"vdemux open \n");
	if ((fd_ademux = open(devnamed, O_RDWR|O_NONBLOCK)) < 0){
	    perror("DMX AUDIO open");
	    return;
	}
	fprintf(stderr,"ademux open \n");

	set_pat_filt(fd);
	if (vpid && apid) set_av_filts(fd_vdemux,fd_ademux
				       ,vpid,apid);

	while ( (count = read(filefd, buf, BUFFY2)) > 0 ){
		written = 0;
		if (av->nopat) {
			set = 0;
			apid_n = 0;
			vpid_n = 0;
			find_bavpids(buf, count, &vpid_n, &apid_n);
			if (apid_n && apid_n != apid){
				apid = apid_n;
				set = 1;
			}
			if (vpid_n && vpid_n != vpid){
				vpid = vpid_n;
				set = 1;
			}
			if (set){
				fprintf(stderr,"reset pids %d %d\n",apid,vpid);
				videoClear(fdv);
				audioClear(fda);
				set_av_filts(fd_vdemux,fd_ademux,vpid,apid);
			}
		}

		if(!ppid){
			ppid = get_pmt_pid(fd);
			if (ppid) set_pmt_filt(fd,ppid);
		} else if (!vpid && !apid){

			napid = new_get_av_pids( fd, &vpid, apids, MAXAPIDS);
			//get_av_pids( fd, &vpid, &apid);
			apid = apids[0];

			fprintf(stderr,"apid: %d  vpid: %d\n",apid, vpid);
			videoStop(fdv);
			audioStop(fda);
			if (vpid && apid) set_av_filts(fd_vdemux,fd_ademux
						       ,vpid,apid);
			videoPlay(fdv);
			audioPlay(fda);
		}
		while( count > 0 && written < count){
			c = write(fd_dvr,buf+written, count-written);
			if (c>0) written += c;
		}
	}
}

static ssize_t my_write(int fd, uint8_t *buf, int count)
{
	return write(fd, buf, count);
}

static void write_out_s(uint8_t *buf, int count, void *priv)
{
	struct pollfd pfd[1];
	ipack *p = (ipack *) priv;
	int written,c;
	int fd = p->fd;
	
	if (fd <0 ) return;

	pfd[0].fd = fd;
	pfd[0].events = POLLOUT;
	
	if(p->playing < 4 && p->fd == p->fd1){
		videoPlay(p->fd1);
		audioPlay(p->fd2);
		p->playing++;  
	}

	written = 0;
	while(written < count){
		if (poll(pfd,1,100)){
			if (pfd[1].revents & POLLOUT){
				if (p->av->needs_demux){
					c = write_demux(fd,buf+written,
							count-written, p->av, 
							my_write);
					if (c>0) written += c;
				} else {
					c = write(fd,buf+written,count-written);
					if (c>0) written += c;
				}
			}
		}
	}
}


void play_ts_video_n(switch_t *sw, uint16_t pida, uint16_t pidv)
{
	uint8_t buf[IN_SIZE];
	uint8_t mbuf[TS_SIZE];
	int count;
	int64_t l = 0;
	int i;
	uint16_t pid;
	uint16_t dummy;
	ipack pa, pv;
	ipack *p;
	int fd = sw->fdv;
	int fda = sw->fda;
	int filefd = sw->filefd;
	av_settings *av = sw->av;
	int fail = 0;

	
	if (videoSetBlank(fd,false) < 0) return;
	if (audioSelectSource(fda,AUDIO_SOURCE_MEMORY) < 0) return;
	if (videoSelectSource(fd,VIDEO_SOURCE_MEMORY) < 0) return;
	if (audioSetAVSync(fda, true) < 0) return;

	init_ipack(&pa, IPACKS, write_out_s,0);
	init_ipack(&pv, IPACKS, write_out_s,0);

	pa.data = (void *)&pa;
	pv.data = (void *)&pv;
	pa.av = av;
	pv.av = av;
	pv.fd   = fd;
	if (av->needs_demux)
		pa.fd   = fda;
	else 
		pa.fd   = fd;
		
	pv.fd1  = fd;
	pa.fd1  = fd;
	pv.fd2  = fda;
	pa.fd2  = fda;
	pv.ffd  = filefd;
	pa.ffd  = filefd;

	count = read(filefd,mbuf,TS_SIZE);
	l+= count;
	for ( i = 0; i < 188 ; i++){
		if ( mbuf[i] == 0x47 ) break;
	}


	if ( i == 188){
		printf("Not a TS\n");
	} else {
		memcpy(buf,mbuf+i,TS_SIZE-i);
		count = read(filefd,mbuf,i);
		l+= count;
		memcpy(buf+TS_SIZE-i,mbuf,i);
		i = 188;
	}

	pv.playing = 1;
	pa.playing = 1;

	while ( pv.playing > 0 && pa.playing > 0 ){
		if ( ( count = read(filefd,buf+i,IN_SIZE-i)+i) < 0){
		        perror("ts play ");
			exit(1);
		}		
		if (count <= 0){
			fail++;
			if (fail > 10)
				break;
		}
		if (!pidv){
			find_bavpids(buf+i, IN_SIZE-i, &pidv, &dummy);
			if (pidv && !sw->stop) 
				printf("vpid %d (0x%02x)\n",pidv,pidv);
		} 

		if (!pida){
			find_bavpids(buf+i, IN_SIZE-i, &dummy, &pida);
			if (pida && !sw->stop) 
				printf("apid %d (0x%02x)\n",pida,pida);
		} 

		if (pida && pidv){
			for( i = 0; i < count; i+= TS_SIZE){
				uint8_t off = 0;

				if ( count - i < TS_SIZE) break;
				
				pid = get_pid(buf+i+1);
				if (!(buf[3+i]&0x10)) // no payload?
					continue;
				if (pid == pidv){
					p = &pv;
				} else {
					if (pid == pida){
						p = &pa;
					} else continue;
				}
				if ( buf[1+i]&0x40) {
					if (p->plength == MMAX_PLENGTH-6){
						p->plength = p->found-6;
						p->found = 0;
						send_ipack(p);
						reset_ipack(p);
					}
				}

				if ( buf[3+i] & 0x20) {  // adaptation field?
					off = buf[4+i] + 1;
				}
				instant_repack(buf+4+off+i, TS_SIZE-4-off, p);
			}
		}
		i = 0;

		if (sw->stop && *sw->stop){
			pv.playing = 0;
			pa.playing = 0;
		}
	}
	if (sw->stop) *sw->stop = 1;
	audioSelectSource(fda,AUDIO_SOURCE_DEMUX);
	videoSelectSource(fd,VIDEO_SOURCE_DEMUX);
	videoPlay(fd);
	audioSetAVSync(fda, true);
}


int seek_pes_start(uint8_t *buf, int size)
{
        int found = 0;
        int c=0;

        while ( found < 4 ){
                while (found != 4){
                        switch (found) {
                        case 0:
                                if ( buf[c] == 0x00 ) found++;
                                c++;
                                break;
                        case 1:
                                if ( buf[c] == 0x00 ) found++;
                                else found = 0;
                                c++;
                                break;                     
                        case 2:
                                if ( buf[c] == 0x01 ) found++;
                                else found = 0;
                                if ( buf[c] == 0x00 ) found = 2;
                                c++;
                                break;
                        
                        case 3:
                                if ( (buf[c] & 0xe0) == 0xe0 ) found++;
                                else found = 0;
                                c++;
                                break;
                        }
                        if (c >= size) return -1;
                }
	}
        return c-4;
}

int check_fd_ts(int fd)
{
	uint8_t buf[CHECKBUF];	
	int mark=0;
	int found = 0;
	
	while( !found ){
		read(fd, buf, CHECKBUF);
		if ( seek_pes_start(buf,CHECKBUF) > 0)
			found=1;
	}
	if (!found)  return C_NOPES;


	while ( mark < CHECKBUF-2*188){
		if (buf[mark] == 0x47 && buf[mark+188] == 0x47
		    && buf[mark+2*188] == 0x47) 
			return C_TS;
		else mark++;
	}
	
	return C_MPEG;
}


int check_file_ts(char *filename)
{
	int fd;
	int ret;
	
	if ( (fd = open(filename,O_RDONLY)) < 0){
		perror("File open:");
		return -1;
	}
	ret = check_fd_ts(fd);
	close(fd);
	return ret;
}



#ifdef PTHREADS

void rec_out(uint8_t *buf, int count,void  *p)
{
	ipack *ip = (ipack *)p;
	poll_write(ip->fd,buf,count);
}

void record_mpeg(void *din)
{
	uint16_t pida;
	uint16_t pidv;
	uint8_t buf[IN_SIZE];
	uint8_t mbuf[TS_SIZE];
	struct pollfd pfd[1];
	int i;
	int count = 0;
	int c = 0;
	uint16_t pid;
	ipack pa, pv;
	ipack *p;
	int status;
	int fd_dvr;
	int len;
	dvr_info_t *dinfo = (dvr_info_t *)din;


	status = pthread_detach(pthread_self());
        if (status) fprintf(stderr, "Thread detach %d \n", status);

	pida = dinfo->pida;
	pidv = dinfo->pidv;
	
	init_ipack(&pa, IPACKS, rec_out, 1);
	init_ipack(&pv, IPACKS, rec_out, 1);
	pa.fd = dinfo->filefd;
	pv.fd = dinfo->filefd;
	pv.data =(void *) &pv;
	pa.data =(void *) &pa;
	dinfo->length = 0;
	dinfo->stop = 0;
	fd_dvr = dinfo->fd_dvr;
	
	pfd[0].fd = fd_dvr;
	pfd[0].events = POLLIN;
	while (count < TS_SIZE){
		if (poll(pfd,1,100)){
			if (pfd[0].revents & POLLIN){
				if ((c = read(fd_dvr, mbuf+count, 
					      TS_SIZE-count)) >0){
					count += c;
				}
			}
		}
	}

	for ( len = 0; len < 188 ; len++){
		if ( mbuf[len] == 0x47 ) break;
	}
	if ( len == 188){
		fprintf(stderr,"Not a TS %d\n",count);
		dinfo->stop = 1;
		close (fd_dvr);
		close (dinfo->filefd);
		pthread_exit(NULL);
		return;
	} else {
		memcpy(buf,mbuf+len,TS_SIZE-len);
		count = 0;
		while (count < len){
			if (poll(pfd,1,100)){
				if (pfd[0].revents & POLLIN){
					if ((c = read(fd_dvr, mbuf+count, 
						      len-count)) >0){
						count += c;
					}
				}
			}
		}
		memcpy(buf+TS_SIZE-len,mbuf,len);
		len = 188;
	}

	dinfo->length = len;
	while (!dinfo->stop){
		count = 0;
		while (count < IN_SIZE-len){
			if (poll(pfd,1,100)){
				if (pfd[0].revents & POLLIN){
					if ((c = read(fd_dvr, buf+len+count, 
						      IN_SIZE-len-count)) >0){
						count += c;
					}
				}
			}
		}
		for( i = 0; i < IN_SIZE; i+= TS_SIZE){
			uint8_t off = 0;

			pid = get_pid(buf+i+1);
			if (!(buf[3+i]&0x10)) // no payload?
				continue;
			if (pid == pidv){
				p = &pv;
			} else {
				if (pid == pida){
					p = &pa;
				} else continue;
			}

			if ( buf[1+i]&0x40) {
				if (p->plength == MMAX_PLENGTH-6){
					p->plength = p->found-6;
					p->found = 0;
					send_ipack(p);
					reset_ipack(p);
				}
			}

			if ( buf[3+i] & 0x20) {  // adaptation field?
				off = buf[4+i] + 1;
			}
			instant_repack(buf+4+off+i, TS_SIZE-4-off, p);
		}
		dinfo->length += IN_SIZE-len;
		len = 0;
	}

	pv.plength = pv.found-6;
	pv.found = 0;
	send_ipack(&pv);
	reset_ipack(&pv);

	pa.plength = pa.found-6;
	pa.found = 0;
	send_ipack(&pa);
	reset_ipack(&pa);

	close (fd_dvr);
	close (dinfo->filefd);
        pthread_exit(NULL);
}



#define DVR_BUFFY 188*20

void record_ts(void *din)
{
	int len;
	int written;
	uint8_t buf[DVR_BUFFY];
	struct pollfd pfd[1];
	int fd_dvr;
	int fd_dvr_out;
	int status;
	uint64_t max_length = 0;
	dvr_info_t *dinfo = (dvr_info_t *)din;

	status = pthread_detach(pthread_self());
        if (status) fprintf(stderr, "Thread detach %d \n", status);

	dinfo->length = 0;
	if (dinfo->stop) max_length = dinfo->stop;
	dinfo->stop = 0;
	
	fd_dvr = dinfo->fd_dvr;
	fd_dvr_out = dinfo->filefd;
	
	pfd[0].fd = fd_dvr;
	pfd[0].events = POLLIN;
	
	while ( !dinfo->stop ) {
		if (poll(pfd,1,1000)){
			if (pfd[0].revents & POLLIN){
				len = read(fd_dvr, buf, DVR_BUFFY);
				if (len < 0){
					perror("recording");
					dinfo->stop = 1;
					close (fd_dvr);
					close (fd_dvr_out);
					pthread_exit(NULL);
					return;
				}
				if (len > 0){
					written = 0;
					while (written < len) 
						written += 
							write (fd_dvr_out, 
							       buf, len);
					dinfo->length += len;
				}
			}
		}
		if (max_length && dinfo->length>max_length) dinfo->stop=1; 
	}
	close (fd_dvr);
	close (fd_dvr_out);
	pthread_exit(NULL);
}

int c_dvr_thread_record(pthread_t *thread,  dvr_info_t *dinfo)
{
	int status = -1;

	switch (dinfo->type){
	case C_TS:
		status = pthread_create( thread, NULL, (void *) &record_ts, 
				       (void *) dinfo);
		break;

	case C_MPEG:
		status = pthread_create( thread, NULL, (void *) &record_mpeg, 
				       (void *) dinfo);
		break;
	}
	if (status){
		fprintf(stderr, "Thread detach %d \n",status);
	} 

	return status;
}
	
void replay_ts(void *din)
{
	switch_t *sw;
	av_settings av;

	dvr_info_t *dinfo = (dvr_info_t *)din;

        if (open_av(&dinfo->fdv, &dinfo->fda, dinfo->dev, 0) < 0){
		perror("open av");
		dinfo->stop = 1;
		close(dinfo->fda);
		close(dinfo->fdv);
		pthread_exit(NULL);
		return;
	}
	

	sw = &dinfo->sw;
	sw->filefd = dinfo->filefd;
	sw->stop = &dinfo->stop;
	sw->ts = 1;
	sw->av = &av;
	sw->fdv = dinfo->fdv;
	sw->fda = dinfo->fda;
	init_av(&av);
	av.nopat = 1;

	next_gop_ts(dinfo->filefd,0);
	play_ts_video_n(sw, 0, 0);
	mute(sw->fda, 0, sw->av);
	sw->fdv = -1;
	sw->fda = -1;
	close(dinfo->fda);
	close(dinfo->fdv);
        pthread_exit(NULL);
}

void replay_mpeg(void *din)
{
	switch_t *sw;
	av_settings av;

	dvr_info_t *dinfo = (dvr_info_t *)din;

        if (open_av(&dinfo->fdv, &dinfo->fda, dinfo->dev, 0) < 0){
		perror("open av");
		dinfo->stop = 1;
		close(dinfo->fda);
		close(dinfo->fdv);
		pthread_exit(NULL);
		return;
	}
	
	sw = &dinfo->sw;
	sw->filefd = dinfo->filefd;
	sw->stop = &dinfo->stop;
	sw->ts = 0;
	sw->av = &av;
	sw->fdv = dinfo->fdv;
	sw->fda = dinfo->fda;
	init_av(&av);
	av.nopat = 1;

//	next_gop(dinfo->filefd,0);
	play_file_video_mpg(sw);
	mute(sw->fda, 0, sw->av);
	sw->fdv = -1;
	sw->fda = -1;
	close(dinfo->fda);
	close(dinfo->fdv);
        pthread_exit(NULL);
}


int c_dvr_thread_play(pthread_t *thread,  dvr_info_t *dinfo)
{
	int status = -1;

	switch (dinfo->type){
	case C_TS:
		status = pthread_create( thread, NULL, (void *) &replay_ts, 
					 (void *) dinfo);
		break;

	case C_MPEG:
		status = pthread_create( thread, NULL, (void *) &replay_mpeg, 
					 (void *) dinfo);
		break;
	}
	if (status){
		fprintf(stderr, "Thread detach %d \n",status);
	} 

	return status;
}



#endif

