#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>

#include <linux/dvb/dmx.h>
#include <linux/dvb/frontend.h>

#include "list.h"
#include "dump.h"

#define VERBOSE 1
#ifdef VERBOSE
#define MSG(x...) 						\
do {								\
	fprintf (stderr, x);					\
	fprintf (stderr, "\n");					\
} while (0)
#else
#define MSG(x...)
#endif

#define WARN(x...) 						\
do {								\
	fprintf (stderr, "WARNING (%s): ", __FUNCTION__);	\
	fprintf (stderr, x);					\
	fprintf (stderr, "\n");					\
} while (0)


#define PERROR(x...) 						\
do {								\
	fprintf (stderr, "%s: ", __FUNCTION__);			\
	perror (x);						\
} while (0)


struct list_entry {
	int id1;
	int id2;
};


struct list {
	char *name;
	struct list_entry *entry;
	int count;
	int entry_size;
};


struct network {
	int dummy;
	int network_id;
	char *network_name;
};


struct transponder {
	int network_id;
	int transport_stream_id;
	fe_type_t type;
	struct dvb_frontend_parameters param;
};


typedef enum {
	ST_TV       = 0x01,
	ST_RADIO    = 0x02,
	ST_TELETEXT = 0x03,
	ST_MHP      = 0x10
} ServiceType;


typedef enum {
	RM_NOT_RUNNING = 0x01,
	RM_STARTS_SOON = 0x02,
	RM_PAUSING     = 0x03,
	RM_RUNNING     = 0x04
} RunningMode;


struct service {
	int transport_stream_id;
	int service_id;
	char *provider_name;
	char *service_name;
	uint16_t pmt_pid;
	uint16_t pcr_pid;
	uint16_t video_pid;
	uint16_t audio_pid;
	int private_pid_count  : 8;
	ServiceType type       : 8;
	int scrambled          : 1;
	RunningMode running    : 3;
	void *priv;
};


static
struct list network_list = {
	"network_list",
	NULL,
	0,
	sizeof(struct network)
};

static
struct list transponder_list = {
	"transponder_list",
	NULL,
	0,
	sizeof(struct transponder)
};

static 
struct list service_list = {
	"service_list",
	NULL,
	0,
	sizeof(struct service)
};



static 
void* find_entry_by_id (struct list *list, int id1, int id2)
{
	int i;
	long entry = (long) list->entry;

	for (i=0; i<list->count; i++, entry += list->entry_size) {
		struct list_entry *e = (struct list_entry*) entry;

		if (e->id1 == id1 && (e->id2 == id2 || id2 == -1))
			return e;

		if (e->id2 == id2 && id1 == -1)
			return e;

		if (e->id1 == id1 && e->id2 == -1) {
			e->id2 = id2;
			return e;
		}

		if (e->id1 == -1 && e->id2 == id2) {
			e->id1 = id1;
			return e;
		}
	}

	return NULL;
}


static 
void* find_or_alloc_entry_by_id (struct list *list, int id1, int id2)
{
	struct list_entry *e = find_entry_by_id (list, id1, id2);
	long addr;

	if (!e) {
		list->entry = realloc (list->entry,
				       (list->count+1) * list->entry_size);

		addr = (long) list->entry + list->count * list->entry_size;
		e = (struct list_entry*) addr;

		memset (e, 0, list->entry_size);

		e->id1 = id1;
		e->id2 = id2;
		list->count++;
	}
	
	return e;
}


static
void* find_nth_entry (struct list *list, unsigned int n)
{
	if (n >= list->count)
		return NULL;

	return (void*) ((long) list->entry + n * list->entry_size);
}


static
void parse_network_name_descriptor (const unsigned char *buf, struct network *n)
{
	unsigned char len = buf [1];

	n->network_name = malloc (len + 1);
	memcpy (n->network_name, buf + 2, len);
	n->network_name[len] = '\0';
	MSG("(%s)", n->network_name);
}


static
long bcd32_to_cpu (const int b0, const int b1, const int b2, const int b3)
{
	return ((b0 >> 4) & 0x0f) * 10000000 + (b0 & 0x0f) * 1000000 +
	       ((b1 >> 4) & 0x0f) * 100000   + (b1 & 0x0f) * 10000 +
	       ((b2 >> 4) & 0x0f) * 1000     + (b2 & 0x0f) * 100 +
	       ((b3 >> 4) & 0x0f) * 10       + (b3 & 0x0f);
}


static const fe_code_rate_t fec_tab [8] = {
	FEC_AUTO, FEC_1_2, FEC_2_3, FEC_3_4,
	FEC_5_6, FEC_7_8, FEC_NONE, FEC_NONE
};


static const fe_modulation_t qam_tab [6] = {
	QAM_AUTO, QAM_16, QAM_32, QAM_64, QAM_128, QAM_256
};


static
void parse_cable_delivery_system_descriptor (const unsigned char *buf,
					     struct transponder *t)
{
	t->type = FE_QAM;

	t->param.frequency = bcd32_to_cpu (buf[2], buf[3], buf[4], buf[5]);
	t->param.frequency *= 100;
	t->param.u.qam.fec_inner = fec_tab[buf[12] & 0x07];
	t->param.u.qam.symbol_rate = bcd32_to_cpu (buf[9], buf[10],
						    buf[11], buf[12] & 0xf0);
	if ((buf[8] & 0x0f) > 5)
		t->param.u.qam.modulation = QAM_AUTO;
	else
		t->param.u.qam.modulation = qam_tab[buf[8] & 0x0f];

	dump_dvb_parameters (stderr, t->type, &t->param);
	fprintf (stderr, "\n");
}


static
void parse_satellite_delivery_system_descriptor (const unsigned char *buf,
						 struct transponder *t)
{
	t->type = FE_QPSK;
	t->param.frequency = bcd32_to_cpu (buf[2], buf[3], buf[4], buf[5]);
	t->param.u.qpsk.fec_inner = fec_tab[buf[12] & 0x07];
	t->param.u.qpsk.symbol_rate = bcd32_to_cpu (buf[9], buf[10],
						    buf[11], buf[12] & 0xf0);

	dump_dvb_parameters (stderr, t->type, &t->param);
	fprintf (stderr, "\n");
}


static
void parse_terrestrial_delivery_system_descriptor (const unsigned char *buf,
						   struct transponder *t)
{
	static const fe_modulation_t m_tab [] = { QPSK, QAM_16, QAM_64, QAM_AUTO };
	static const fe_code_rate_t ofec_tab [8] = { FEC_1_2, FEC_2_3, FEC_3_4,
					       FEC_5_6, FEC_7_8 };
	struct dvb_ofdm_parameters *o = &t->param.u.ofdm;
	
	t->type = FE_OFDM;

	t->param.frequency = (buf[2] << 24) | (buf[3] << 16);
	t->param.frequency |= (buf[4] << 8) | buf[5];
	t->param.frequency *= 10;

	o->bandwidth = BANDWIDTH_8_MHZ + ((buf[6] >> 5) & 0x3);
	o->constellation = m_tab[(buf[7] >> 6) & 0x3];
	o->hierarchy_information = HIERARCHY_NONE + ((buf[7] >> 3) & 0x3);

	if ((buf[7] & 0x7) > 4)
		o->code_rate_HP = FEC_AUTO;
	else
		o->code_rate_HP = ofec_tab [buf[7] & 0x7];

	if (((buf[8] >> 5) & 0x7) > 4)
		o->code_rate_HP = FEC_AUTO;
	else
		o->code_rate_HP = ofec_tab [(buf[8] >> 5) & 0x7];

	o->guard_interval = GUARD_INTERVAL_1_32 + ((buf[8] >> 3) & 0x3);

	o->transmission_mode = (buf[8] & 0x2) ?
			       TRANSMISSION_MODE_8K :
			       TRANSMISSION_MODE_2K;

	dump_dvb_parameters (stderr, t->type, &t->param);
	fprintf (stderr, "\n");
}


static
void parse_service_descriptor (const unsigned char *buf, struct service *s)
{
	unsigned char len;

	s->type = buf[2];

	buf += 3;
	len = *buf;
	buf++;

	if (s->provider_name)
		free (s->provider_name);

	s->provider_name = malloc (len + 1);
	memcpy (s->provider_name, buf, len);
	s->provider_name[len] = '\0';

	if (s->service_name)
		free (s->service_name);

	buf += len;
	len = *buf;
	buf++;
	s->service_name = malloc (len + 1);
	memcpy (s->service_name, buf, len);
	s->service_name[len] = '\0';

	MSG("0x%04x 0x%04x: pmt_pid 0x%04x %s -- %s (%s, %sscrambled)",
	    s->transport_stream_id,
	    s->service_id,
	    s->pmt_pid,
	    s->provider_name, s->service_name,
	    s->running == RM_NOT_RUNNING ? "not running" :
	    s->running == RM_STARTS_SOON ? "starts soon" :
	    s->running == RM_PAUSING     ? "pausing" :
	    s->running == RM_RUNNING     ? "running" : "???",
	    s->scrambled ? "" : "not ");
}


static
void parse_descriptors (const unsigned char *buf, int descriptors_loop_len,
			void *data)
{
	while (descriptors_loop_len > 0) {
		unsigned char descriptor_tag = buf[0];
		unsigned char descriptor_len = buf[1] + 2;

		if (!descriptor_len) {
			WARN("descriptor_tag == 0x%02x, descriptor_len == %i",
			     descriptor_tag, descriptor_len);
			break;
		}

		switch (descriptor_tag) {
		case 0x40:
			parse_network_name_descriptor (buf, data);
			break;

		case 0x43:
			parse_satellite_delivery_system_descriptor (buf, data);
			break;

		case 0x44:
			parse_cable_delivery_system_descriptor (buf, data);
			break;

		case 0x48:
			parse_service_descriptor (buf, data);
			break;

		case 0x5a:
			parse_terrestrial_delivery_system_descriptor (buf, data);
			break;

		default:
			/*WARN("skip descriptor 0x%02x", descriptor_tag)*/;
		};

		buf += descriptor_len;
		descriptors_loop_len -= descriptor_len;
	}
}


static
void parse_pat (const unsigned char *buf, int section_length,
		int transport_stream_id)
{
	buf += 4;		/*  skip nit pid entry... */
	section_length -= 4;

	while (section_length > 0) {
		struct service *s;
		int service_id = (buf[0] << 8) | buf[1];

		s = find_or_alloc_entry_by_id (&service_list,
					       transport_stream_id,
					       service_id);

		s->pmt_pid = ((buf[2] & 0x1f) << 8) | buf[3];

		buf += 4;
		section_length -= 4;
	};
}


static
void parse_pmt (const unsigned char *buf, int section_length, int service_id)
{
	int program_info_len;
	struct service *s;

	s = find_or_alloc_entry_by_id (&service_list,
				       -1,
				       service_id);

	s->pcr_pid = ((buf[0] & 0x1f) << 8) | buf[1];

	program_info_len = ((buf[2] & 0x0f) << 8) | buf[3];

	buf += program_info_len + 4;
	section_length -= program_info_len + 4;

	while (section_length > 0) {
		int ES_info_len = ((buf[3] & 0x0f) << 8) | buf[4];
		int elementary_pid = ((buf[1] & 0x1f) << 8) | buf[2];

		switch (buf[0]) {
		case 0x01:
		case 0x02:
			s->video_pid = elementary_pid;
			break;
		case 0x03:
		case 0x04:
			s->audio_pid = elementary_pid;
			break;
		default:
			s->private_pid_count++;
		};

		buf += ES_info_len + 5;
		section_length -= ES_info_len + 5;
	};
}


static
void parse_nit (const unsigned char *buf, int section_length, int network_id)
{
	int descriptors_loop_len = ((buf[0] & 0x0f) << 8) | buf[1];
	struct network *n;

	n = find_or_alloc_entry_by_id (&network_list, 0, network_id);

	parse_descriptors (buf + 2, descriptors_loop_len, n);

	section_length -= descriptors_loop_len + 4;
	buf += descriptors_loop_len + 4;

	while (section_length > 4) {
		int transport_stream_id = (buf[0] << 8) | buf[1];
		struct transponder *t;

		descriptors_loop_len = ((buf[4] & 0x0f) << 8) | buf[5];

		if (!transport_stream_id || !descriptors_loop_len) {
			WARN("service_id == 0x%02x, "
			     "descriptors_loop_len == %i",
			     transport_stream_id, descriptors_loop_len);
			break;
		}

		t = find_or_alloc_entry_by_id (&transponder_list,
					       network_id,
					       transport_stream_id);

		parse_descriptors (buf + 6, descriptors_loop_len, t);

		section_length -= descriptors_loop_len + 6;
		buf += descriptors_loop_len + 6;
	};
}


static
void parse_sdt (const unsigned char *buf, int section_length,
		int transport_stream_id)
{
	buf += 3;               /*  skip original network id + reserved field */

	while (section_length > 4) {
		int service_id = (buf[0] << 8) | buf[1];
		int descriptors_loop_len = ((buf[3] & 0x0f) << 8) | buf[4];
		struct service *s;

		if (!service_id || !descriptors_loop_len) {
			WARN("service_id == 0x%02x, "
			     "descriptors_loop_len == %i",
			     service_id, descriptors_loop_len);
			break;
		}

		s = find_or_alloc_entry_by_id (&service_list,
					       transport_stream_id,
					       service_id);

		s->running = (buf[3] >> 5) & 0x7;
		s->scrambled = (buf[3] >> 4) & 1;

		parse_descriptors (buf + 5, descriptors_loop_len, s);

		section_length -= descriptors_loop_len + 5;
		buf += descriptors_loop_len + 5;
	};
}


struct section_buf {
	struct list_head list_head;
	const char *dmx_device;
	int run_once;
	int fd;
	int pid;
	int table_id [3];
	int id [3];
	int section_version_number [3];
	int start_section_number [3];
	int section_done [3];
	unsigned char buf [1024];
	int bytes_in_buf;
	time_t timeout;
	time_t start_time;
	time_t running_time;
};


/**
 *   returns 0 when more sections are expected
 *           1 when all sections are read
 *           -1 on invalid table id
 */
static
int parse_section (struct section_buf *s)
{
	const unsigned char *buf = s->buf;
	int table_id;
	int section_length;
	int id;
	int section_version_number;
	int section_number;
	int last_section_number;
	int i;

	table_id = buf[0];

	for (i=0; i<3; i++)
		if (s->table_id[i] == table_id)
			break;

	if (i > 2)
		return -1;

	section_length = (((buf[1] & 0x0f) << 8) | buf[2]) - 11;

	id = (buf[3] << 8) | buf[4];
	section_version_number = (buf[5] >> 1) & 0x1f;
	section_number = buf[6];
	last_section_number = buf[7];

	MSG("pid 0x%02x tid 0x%02x id 0x%04x, %i/%i (version %i)",
	    s->pid, table_id, id, section_number,
	    last_section_number, section_version_number);

	buf += 8;

	if (s->section_version_number[i] != section_version_number ||
	    s->id[i] != id)
	{
		s->id[i] = id;
		s->section_version_number[i] = section_version_number;
		s->start_section_number[i] = section_number;
		if (section_number == 0 && last_section_number == 0)
			s->section_done[i] = 1;
		else
			s->section_done[i] = 0;
	} else {
		if (s->start_section_number[i] == section_number)
			s->section_done[i] = 1;
	}

	switch (table_id) {
	case 0x00:
		parse_pat (buf, section_length, id);
		break;

	case 0x02:
		parse_pmt (buf, section_length, id);
		break;

	case 0x40:
	case 0x41:
		parse_nit (buf, section_length, id);
		break;

	case 0x42:
	case 0x46:
		parse_sdt (buf, section_length, id);
		break;

	default:
		;
	};

	if (s->start_section_number[i] == 0 && section_number == last_section_number)
		s->section_done[i] = 1;

	if (s->section_done[0] && s->section_done[1] && s->section_done[2])
		return 1;

	return 0;
}


static
int read_sections (struct section_buf *s)
{
	int section_length, count;

	if (s->section_done[0] && s->section_done[1] && s->section_done[2])
		return 1;

	if ((count = read (s->fd, s->buf, 1024 - s->bytes_in_buf)) < 0)
		if ((count = read (s->fd, s->buf, 1024 - s->bytes_in_buf)) < 0)
			return -1;

	s->bytes_in_buf += count;

	if (s->bytes_in_buf >= 1024)
		s->bytes_in_buf = 0;

	if (s->bytes_in_buf < 4)
		return -1;

	section_length = ((s->buf[1] & 0x0f) << 8) | s->buf[2];

	if (s->bytes_in_buf > section_length + 3)
		s->bytes_in_buf = 0;

	if (section_length == s->bytes_in_buf - 3)
		if (parse_section (s) == 1)
			return 1;

	return 0;
}


static LIST_HEAD(running_filters);
static LIST_HEAD(waiting_filters);


static
void setup_filter (struct section_buf* s, const char *dmx_device,
		   int pid, int tid0, int tid1, int tid2,
		   int run_once, int timeout)
{
	int i;

	memset (s, 0, sizeof(struct section_buf));

	s->dmx_device = dmx_device;
	s->pid = pid;
	s->table_id[0] = tid0;
	s->table_id[1] = tid1;
	s->table_id[2] = tid2;

	s->run_once = run_once;
	s->timeout = timeout;

	for (i=0; i<3; i++) {
		s->id[i] = -1;
		s->section_version_number[i] = -1;
		s->start_section_number[i] = -1;
	}

	INIT_LIST_HEAD (&s->list_head);
}


static
int start_filter (struct section_buf* s)
{
	struct dmx_sct_filter_params f;
	int i;

	s->bytes_in_buf = 0;

	if ((s->fd = open (s->dmx_device, O_RDWR | O_NONBLOCK)) < 0) {
		PERROR ("open");
		goto err0;
	}

	memset(&f, 0, sizeof(struct dmx_sct_filter_params));

	f.pid = (uint16_t) s->pid;

	if (s->table_id[1] == -1 && s->table_id[2] == -1 &&
	    s->table_id[0] < 0x100 && s->table_id[0] > 0)
	{
       	        f.filter.filter[0] = (uint8_t) s->table_id[0];
               	f.filter.mask[0]   = 0xff;
	}

	for (i=0; i<3; i++)
		if (s->table_id[i] == -1)
			s->section_done[i] = 1;
		else
			s->section_done[i] = 0;

        f.timeout = 0;
        f.flags = DMX_IMMEDIATE_START;

	if (ioctl(s->fd, DMX_SET_FILTER, &f) == -1) {
                PERROR ("ioctl DMX_SET_FILTER");
		goto err1;
	}

	time (&s->start_time);

	list_add (&s->list_head, &running_filters);

	return 0;

err1:
	ioctl (s->fd, DMX_STOP);
	close (s->fd);
err0:
	return -1;
}


static
void stop_filter (struct section_buf *s)
{
	ioctl (s->fd, DMX_STOP);
	close (s->fd);
	list_del (&s->list_head);
	s->running_time += time(NULL) - s->start_time;
}


void add_filter (struct section_buf *s)
{
	if (start_filter (s))
		list_add_tail (&s->list_head, &waiting_filters);
}


void remove_filter (struct section_buf *s)
{
	stop_filter (s);

	while (!list_empty(&waiting_filters)) {
		struct list_head *next = waiting_filters.next;
		s = list_entry (next, struct section_buf, list_head);
		if (start_filter (s))
			break;
	};
}


void reschedule_filter (struct section_buf *s)
{
	remove_filter (s);
	add_filter (s);
}


void read_filters (void)
{
	struct list_head *p, *n;

	list_for_each_safe (p, n, &running_filters) {
		struct section_buf *s;
		int done;

		s = list_entry (p, struct section_buf, list_head);

		done = read_sections (s) == 1;

//printf ("pid 0x%04x, tid %02x/%02x/%02x, running %i sec -- %s\n", s->pid, s->table_id[0], s->table_id[1], s->table_id[2], time(NULL) - s->start_time, done ? "done" : "");

		if (done || time(NULL) > s->start_time + s->timeout) {
			if (s->run_once)
				remove_filter (s);
			else
				reschedule_filter(s);
		}
	}
}


int main ()
{
	struct section_buf s0;
	struct section_buf s1;
	struct section_buf s2;
	int start_time = time(NULL);
	int i;

	setup_filter (&s0, "/dev/dvb/adapter0/demux0", 0x00, 0x00, -1, -1, 1, 10);
	setup_filter (&s1, "/dev/dvb/adapter0/demux0", 0x10, 0x40, -1, -1, 1, 20);
	setup_filter (&s2, "/dev/dvb/adapter0/demux0", 0x11, 0x42, -1, -1, 1, 20);

	add_filter (&s0);
	add_filter (&s1);
	add_filter (&s2);

	do {
		read_filters ();

		for (i=0; i<service_list.count; i++) {
			struct service *s = find_nth_entry (&service_list, i);

			if (!s->priv && s->pmt_pid) {
				s->priv = malloc (sizeof(struct section_buf));
				setup_filter (s->priv,
					      "/dev/dvb/adapter0/demux0",
					      s->pmt_pid, 0x02, -1, -1, 1, 10);
				add_filter (s->priv);
			}
		}
	} while (time(NULL) < start_time + 20);

	for (i=0; i<service_list.count; i++) {
		struct service *s = find_nth_entry (&service_list, i);
		struct transponder *t;
		struct network *n;

		t = find_entry_by_id (&transponder_list, -1,
				      s->transport_stream_id);

		if (t) {
			n = find_entry_by_id (&network_list, -1,
					      t->network_id);

			dump_service_parameter_set (stdout,
						    s->service_name,
						    t->type,
						    &t->param,
						    s->video_pid,
						    s->audio_pid);
		}
	}

	return 0;
}


